/***************************************************************************
 *   Copyright (C) 2009 by Paul Lutus                                      *
 *   lutusp@arachnoid.com                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
package magiclantern;

import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.ClipboardOwner;
import java.awt.*;
import java.io.*;
import javax.imageio.*;
import java.awt.image.*;

import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.dnd.DragSourceContext;

import java.awt.datatransfer.StringSelection;
import java.awt.dnd.DnDConstants;
import java.awt.dnd.DragGestureEvent;
import java.awt.dnd.DragGestureListener;
import java.awt.dnd.DragSource;
import java.awt.dnd.DragSourceDragEvent;
import java.awt.dnd.DragSourceDropEvent;
import java.awt.dnd.DragSourceEvent;
import java.awt.dnd.DragSourceListener;
import java.util.*;



import java.awt.event.*;
import javax.swing.*;

/**
 *
 * @author lutusp
 */
public class EditBrowsePanelCoreRoutines extends javax.swing.JPanel implements ClipboardOwner, DragGestureListener, DragSourceListener {

    MyDefaultListModel editListModel;
    JList editList = null;
    JButton saveButton;
    JPopupMenu editPopupMenu = null;
    MagicLantern parent;
    EditBase base;
    DragSource dragSource;
    Object[] draggedItems;
    Stack<Object[]> undoStack;
    Stack<Object[]> redoStack;
    //boolean enabled = false;
    boolean contentChanged = false;
    String activeFile = null;

    public EditBrowsePanelCoreRoutines(MagicLantern p, EditBase b) {
        parent = p;
        base = b;
        resetUndoRedo();
    }

    void setupTooltips(boolean create) {
        String showTooltip = null;
        String browseTooltip = null;
        if (create) {
            showTooltip = "<html>This panel is for creating and editing<br/>";
            showTooltip += "your slide show. To create a new show,<br/>";
            showTooltip += "press the \"Create\" button at the upper right.</html>";
            browseTooltip = "<html>This panel is a scratch-pad area<br/>";
            browseTooltip += "for temporary storage of images<br/>";
            browseTooltip += "you might want to add to your slide show.</html>";

        }
        final String stt = showTooltip;
        final String btt = browseTooltip;
        SwingUtilities.invokeLater(
                new Runnable() {

                    public void run() {
                        base.editPanel.getList().setToolTipText(stt);
                        base.browsePanel.getList().setToolTipText(btt);
                    }
                });
    }

    public void openFileMessage() {
        CommonCode.beep();
        JOptionPane.showMessageDialog(this, "Cannot edit -- please open a new or existing show", parent.programName + ": No open show", JOptionPane.WARNING_MESSAGE);
    }

    void setEditRenderer() {
        Color bc, fc;
        if (parent.programValues.editWhiteBackground) {
            fc = Color.black;
            bc = Color.white;
        } else {
            bc = Color.black;
            fc = Color.white;
        }
        editList.setCellRenderer(new EditCellRenderer(parent, bc, fc));
        editList.setBackground(bc);
    }

    void initialize(JList el, JPopupMenu popup, JButton saveButton) {
        editList = el;
        this.saveButton = saveButton;
        editPopupMenu = popup;
        editListModel = new MyDefaultListModel();
        editList.setModel(editListModel);
        setEditRenderer();

        editList.addMouseListener(new java.awt.event.MouseAdapter() {

            @Override
            public void mousePressed(java.awt.event.MouseEvent evt) {
                handleMouseEvent(evt);
            }

            @Override
            public void mouseReleased(java.awt.event.MouseEvent evt) {
                handleMouseEvent(evt);
            }

            @Override
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                handleMouseClick(evt);
            }
        });

        editList.setDragEnabled(true);

        editList.setDropMode(DropMode.INSERT);

        editList.setTransferHandler(createTransferHandler(this));

        // Use the default DragSource
        dragSource = DragSource.getDefaultDragSource();

        // Create a DragGestureRecognizer and
        // register as the listener
        dragSource.createDefaultDragGestureRecognizer(editList, DnDConstants.ACTION_COPY_OR_MOVE, this);
    }

    TransferHandler createTransferHandler(final EditBrowsePanelCoreRoutines ep) {
        return new TransferHandler() {

            @Override
            public boolean canImport(TransferHandler.TransferSupport support) {
                boolean result = false;
                try {
                    if (support.isDataFlavorSupported(DataFlavor.stringFlavor) || support.isDataFlavorSupported(DataFlavor.javaFileListFlavor)) {
                        JList.DropLocation dl = (JList.DropLocation) support.getDropLocation();
                        if (dl != null && dl.getIndex() != -1) {
                            result = true;
                        }
                    }
                } catch (Exception e) {
                    System.out.println(e);
                }
                return result;
            }

            @Override
            public boolean importData(TransferHandler.TransferSupport support) {
                //System.out.println("import: [" + support + "]");
                if (activeFile == null) {
                    openFileMessage();
                    return false;
                }
                if (!canImport(support)) {
                    return false;
                }
                Transferable contents = support.getTransferable();
                String data = getTransferableContents(contents);
                if (data == null) {
                    return false;
                }
                pushUndo();
                String array[] = data.split("\n");
                for (int i = array.length - 1; i >= 0; i--) {
                    String path = array[i];
                    // strings pasted from other environments
                    // tend to have junk whitespace
                    path = CommonCode.strip(path);
                    // also the URI prefix that might be present
                    path = path.replaceFirst("file://", "");
                    File f = new File(path);
                    if (f.exists()) {
                        EditCellDatum ecd = new EditCellDatum(parent, ep, path);
                        if (ecd.valid()) {
                            JList.DropLocation dl = (JList.DropLocation) support.getDropLocation();
                            int index = dl.getIndex();
                            if (dl.isInsert()) {
                                editListModel.add(index, ecd);
                            } else {
                                editListModel.set(index, ecd);
                            }
                            setContentChanged(true);
                            Rectangle r = editList.getCellBounds(index, index);
                            scrollRectToVisible(r);
                        }
                    }
                }
                reformatDisplay();
                parent.setStatus("OK", false, 1);
                return true;
            }
        };
    }

    void resetUndoRedo() {
        undoStack = new Stack<Object[]>();
        redoStack = new Stack<Object[]>();
    }

    void populateControl(String path, String[] array) {
        //System.out.println("path: " + path + ", array[0]: " + array[0]);
        if (array != null) {
            editListModel.removeAllElements();
            Arrays.sort(array, new StringComparatorNoCase());
            for (int i = 0; i < array.length; i++) {
                String fn = CommonCode.strip(array[i]);
                //System.out.println(fn);
                //System.out.println("reading: " + array[i]);
                addListItem(path + parent.fileSep + fn);
            }
            reformatDisplay();
        }
    }

    

    // This appears to be the only way to make the calling menu disappear
    // on launching the dialog
    void showImageInfo() {
        Thread t = new Thread() {

            @Override
            public void run() {
                showImageInfoThread();
            }
        };
        t.start();
    }

    void showImageInfoThread() {
        String result = "";
        int index = editList.getSelectedIndex();
        EditCellDatum ecd = (EditCellDatum) editList.getSelectedValue();
        if (ecd != null) {
            try {
                String path = ecd.getPath();
                File f = new File(path);
                BufferedImage im = parent.readImage(f);
                int w = im.getWidth(this);
                int h = im.getHeight(this);
                int type = im.getType();
                int bytesPerPixel = (type == BufferedImage.TYPE_4BYTE_ABGR || type == BufferedImage.TYPE_4BYTE_ABGR_PRE) ? 4 : 3;
                long usize = w * h * bytesPerPixel;
                Date date = new Date(f.lastModified());
                String name = f.getName();
                String pd = f.getParent();
                long size = f.length();
                double aspect = ((double) w / h) + .005;
                aspect = (aspect < 1.0) ? 1.0 / aspect : aspect;
                result = "Index             : " + (index + 1) + "\n";
                result += "Name              : " + name + "\n";
                result += "Location          : " + pd + "\n";
                result += "Last Modified     : " + date + "\n";
                result += "Storage Size      : " + CommonCode.createDecimalFormat(size) + " bytes\n";
                result += "Uncompressed Size : " + CommonCode.createDecimalFormat(usize) + " bytes\n";
                result += "Width             : " + CommonCode.createDecimalFormat(w) + " pixels\n";
                result += "Height            : " + CommonCode.createDecimalFormat(h) + " pixels\n";
                result += "Aspect Ratio      : " + CommonCode.createDecimalFormat(aspect) + "\n";
                String title = parent.programName + ": File Info";
                new MyMessageDialog(parent, true, title, result, parent.getIconImage());

            } catch (Exception e) {
                System.out.println(e);
                result = "Error: Cannot read this image.";
            }
        }
    }

    public DefaultListModel getListModel() {
        return editListModel;
    }

    public JList getList() {
        return editList;
    }

    void MoveFileListToControl(String spath) {
        File flist = new File(spath);
        String array[] = flist.list();
        populateControl(spath, array);
    }

    // this is meant to try to control the JList
    // horizontal wrap mode so it displays the user's
    // choice of columns regardless of the number
    // of slides in the set
    void reformatDisplay() {
        int n = editListModel.size();
        n = (n > 0) ? n - 1 : n;
        int rc = (n / parent.editColumns) + 1;
        editList.setVisibleRowCount(rc);
    }

    void pushUndo() {
        undoStack.push(editListModel.toArray());
        updateButtons();
    }

    void pushRedo() {
        redoStack.push(editListModel.toArray());
        updateButtons();
    }

    void undo() {
        if (undoStack.size() > 0) {
            pushRedo();
            Object[] array = undoStack.pop();
            replaceModelContent(array);
            updateButtons();
        } else {
            CommonCode.beep();
        }
    }

    void redo() {
        if (redoStack.size() > 0) {
            pushUndo();
            Object[] array = redoStack.pop();
            replaceModelContent(array);
            updateButtons();
        } else {
            CommonCode.beep();
        }
    }

    // don't delete these -- they need to be present
    // in order to be overridden in the derivative classes
    void updateButtons() {
    }

    void updateShowDisplay() {
    }

    void setContentChanged(boolean state) {
        contentChanged = state;
        if (saveButton != null) {
            saveButton.setEnabled(contentChanged);
        }
        updateShowDisplay();
        setupTooltips(false);
    }

    void replaceModelContent(Object[] array) {
        editListModel.removeAllElements();
        for (int i = 0; i < array.length; i++) {
            editListModel.addElement(array[i]);
        }
        reformatDisplay();
        setContentChanged(true);
    }

    /*
     * begin clipboard-related methods
     *
     */
    void addListItem(String path) {
        EditCellDatum ecd = new EditCellDatum(parent, this, path);
        if (ecd.valid()) {
            editListModel.addElement(ecd);
            setContentChanged(true);
        }
    }

    void updateCutCopyPasteButtons(JMenuItem mCut, JMenuItem mCopy, JMenuItem mPaste, JButton cut, JButton copy, JButton paste) {
        boolean cutCopyValid = cutCopyActionValid();
        boolean pasteValid = pasteActionValid();
        mCut.setEnabled(cutCopyValid);
        mCopy.setEnabled(cutCopyValid);
        cut.setEnabled(cutCopyValid);
        copy.setEnabled(cutCopyValid);
        mPaste.setEnabled(pasteValid);
        paste.setEnabled(pasteValid);
    }

    boolean cutCopyActionValid() {
        Object[] ecd = editList.getSelectedValues();
        return (ecd != null && ecd.length > 0);
    }

    boolean pasteActionValid() {
        return (getClipboardContents() != null);
    }

    void cutAction() {
        if (activeFile == null) {
            openFileMessage();
            return;
        }
        if (cutCopyActionValid()) {
            Object[] ecd = editList.getSelectedValues();
            pushUndo();
            StringBuffer data = new StringBuffer();
            for (int i = 0; i < ecd.length; i++) {
                data.append(((EditCellDatum) ecd[i]).getPath() + "\n");
                editListModel.removeElement(ecd[i]);
                setContentChanged(true);
            }
            setClipboardContents(data.toString());
            reformatDisplay();
        } else {
            CommonCode.beep();
        }
    }

    String getCopyContent() {
        String result = null;
        Object[] ecd = editList.getSelectedValues();
        if (ecd != null) {
            StringBuffer data = new StringBuffer();
            for (int i = 0; i < ecd.length; i++) {
                data.append(((EditCellDatum) ecd[i]).getPath() + "\n");
            }
            result = data.toString();
        }
        return result;
    }

    void copyAction() {
        if (activeFile == null) {
            openFileMessage();
            return;
        }
        String data = getCopyContent();
        if (data != null) {
            setClipboardContents(data.toString());
        } else {
            CommonCode.beep();
        }
    }

    void pasteAction() {
        if (activeFile == null) {
            openFileMessage();
            return;
        }
        String s = getClipboardContents();
        putPasteContent(s);

    }

    void putPasteContent(String s) {
        pushUndo();
        String array[] = s.split("\n");
        for (int i = array.length - 1; i >= 0; i--) {
            String path = array[i];
            File f = new File(path);
            if (f.exists()) {
                EditCellDatum ecd = new EditCellDatum(parent, this, path);
                int p = editList.getSelectedIndex();
                if (p == -1) {
                    editListModel.addElement(ecd);
                } else {
                    editListModel.add(p, ecd);
                }
                setContentChanged(true);
            }
        }
        reformatDisplay();
    }

    public void lostOwnership(Clipboard aClipboard, Transferable aContents) {
        //do nothing
    }

    void setClipboardContents(String s) {
        StringSelection ss = new StringSelection(s);
        Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
        clipboard.setContents(ss, this);
    }

    String getTransferableContents(Transferable contents) {
        String result = null;
        try {
            if (contents != null) {
                if (contents.isDataFlavorSupported(DataFlavor.stringFlavor)) {
                    result = (String) contents.getTransferData(DataFlavor.stringFlavor);
                } else if (contents.isDataFlavorSupported(DataFlavor.javaFileListFlavor)) {
                    java.util.List fileList = (java.util.List) contents.getTransferData(DataFlavor.javaFileListFlavor);
                    StringBuffer sb = new StringBuffer();
                    for (int i = 0; i < fileList.size(); i++) {
                        sb.append(((String) ((File) fileList.get(i)).getPath()) + "\n");
                    }
                    result = sb.toString();
                } else {
                    //System.out.println("transferable: no supported types.");
                }
            }
        } catch (Exception e) {
            System.out.println("getTransferableContents: " + e);
        }
        return result;
    }

    String getClipboardContents() {
        Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
        Transferable contents = clipboard.getContents(null);
        return getTransferableContents(contents);
    }

    /*
     *
     * begin drag & drop routines
     *
     */
    public void dragGestureRecognized(DragGestureEvent event) {
        //System.out.println("Drag Gesture Recognized!");
        try {
            // preserve list for possible later deletion
            draggedItems = editList.getSelectedValues();
            // get copy data
            String data = getCopyContent();
            if (data != null) {
                StringSelection transferable = new StringSelection(data);
                event.startDrag(DragSource.DefaultMoveDrop, transferable, this);
            }
        } catch (Exception idoe) {
            System.err.println(idoe);
        }
    }

    public void dragEnter(DragSourceDragEvent event) {
    }

    public void dragExit(DragSourceEvent dse) {
    }

    public void dragOver(DragSourceDragEvent dsde) {
    }

    public void dragDropEnd(DragSourceDropEvent event) {
        if (event.getDropSuccess()) {
            int action = event.getDropAction();
            if (action == DnDConstants.ACTION_MOVE) {
                pushUndo();
                for (int i = 0; i < draggedItems.length; i++) {
                    editListModel.removeElement(draggedItems[i]);
                }
                setContentChanged(true);
            }
            reformatDisplay();
        }
    }

    // change cursors if user action changes
    public void dropActionChanged(DragSourceDragEvent event) {
        DragSourceContext ctx = event.getDragSourceContext();
        int action = event.getUserAction();
        if ((action & DnDConstants.ACTION_MOVE) != 0) {
            ctx.setCursor(DragSource.DefaultMoveDrop);
        } else if ((action & DnDConstants.ACTION_COPY) != 0) {
            ctx.setCursor(DragSource.DefaultCopyDrop);
        } else {
            ctx.setCursor(DragSource.DefaultCopyNoDrop);
        }
    }

    void handleMouseEvent(MouseEvent evt) {
        try {
            if (evt.isPopupTrigger()) {
                int x = evt.getX();
                int y = evt.getY();
                int[] indices = editList.getSelectedIndices();
                if (indices == null || indices.length <= 1) {
                    int index = editList.locationToIndex(evt.getPoint());
                    editList.setSelectedIndex(index);
                }
                editPopupMenu.show(evt.getComponent(), x, y);
            }

        } catch (Exception e) {
            System.out.println(e);
        }
    }

    // detect double-click for large preview
    void handleMouseClick(MouseEvent evt) {
        if (evt.getClickCount() == 2) {
            this.setCursor(new Cursor(Cursor.WAIT_CURSOR));
            EditCellDatum ecd = (EditCellDatum) editList.getSelectedValue();
            parent.editPreview(ecd.getFile());
            this.setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
        }
    }
}
