/***************************************************************************
 *   Copyright (C) 2009 by Paul Lutus                                      *
 *   lutusp@arachnoid.com                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
 
package magiclantern;

import java.io.*;
import java.awt.*;
import javax.imageio.*;
import java.awt.image.*;
import java.util.*;

/**
 *
 * @author lutusp
 */
final public class ImagePanel extends javax.swing.JPanel {

    MagicLantern parent;
    String id;
    String currentImagePath = "";
    BufferedImage image = null, scaledImage = null, oldScaledImage = null;
    int width = -1, height = -1;
    int offsetw, offseth;
    int oldOffsetw, oldOffseth;
    boolean testPattern = false;
    float alpha;
    float alphaInv;
    int fadeSteps = -1;
    int fadeDuration = -1;
    Vector<Float> alphaValues;
    boolean fadeMode;
    boolean threadRunning = false;

    /** Creates new form ImagePanel */
    public ImagePanel(MagicLantern p, String id) {
        this.id = id;
        parent = p;
        initComponents();
    }

    public void newImage(int n, String path, boolean force) {
        fadeMode = parent.programValues.fadeMode;
        if (path == null || path.length() == 0) {
            image = null;
        } else if ((!path.equals(currentImagePath) || force)) {
            try {
                image = parent.readImage(path);
                // force recreation of the scaled image
                width = -1;
                height = -1;
                parent.performExtension(path);
                oldScaledImage = scaledImage;
                oldOffsetw = offsetw;
                oldOffseth = offseth;
            } catch (Exception e) {
                System.out.println("newImage(): " + e + " : " + path);
            }
        }
        if (!path.equals(currentImagePath) && fadeMode) {
            // if setup is needed
            if ((fadeSteps != parent.programValues.fadeSteps) || (fadeDuration != parent.programValues.fadeDurationMilliseconds)) {
                fadeSteps = parent.programValues.fadeSteps;
                fadeDuration = parent.programValues.fadeDurationMilliseconds;
                alphaValues = new Vector<Float>();
                for (int i = 0; i <= fadeSteps; i++) {
                    // create an optimal alpha value set
                    // for transition between images:
                    // empirically, the third root of the
                    // linear scale seems to produce the best
                    // overall result
                    float y = (float) Math.pow((double) i / fadeSteps,0.333);
                    alphaValues.add(y);
                }
            }
            if (!threadRunning) {
                Thread t = new Thread() {

                    public void run() {
                        repaintThread();
                    }
                };
                t.start();
            }
        } else { // no fade, just repaint as usual
            repaint();
        }
        currentImagePath = path;
    }

    // nice alpha fade, but requires very fast machine
    void repaintThread() {
        threadRunning = true;
        int timeStep = fadeDuration / fadeSteps;
        try {
            for (int i = 0; i <= fadeSteps; i++) {
                alpha = alphaValues.get(i);
                alphaInv = alphaValues.get(fadeSteps - i);
                repaint();
                Thread.sleep(timeStep);
            }
        } catch (Exception e) {
            System.out.println(e);
        }
        threadRunning = false;
    }

    public void clearImage() {
        image = null;
        currentImagePath = "";
    }

    public void toggleTestPattern() {
        testPattern = !testPattern;
        repaint();
    }

    @Override
    public void paintComponent(Graphics g) {
        //CommonCode.p("paintcomponent");
        super.paintComponent(g);
        Graphics2D g2 = (Graphics2D) g;
        int w = getWidth();
        int h = getHeight();
        if (image != null) {
            // create new scaled image as required --
            // must be located here in the event of a frame resize
            if (w != width || h != height) {
                width = w;
                height = h;
                int iw = image.getWidth(this);
                int ih = image.getHeight(this);
                // create an image that just fits onto the display
                // by scaling to the screen's height or width
                double wr = (double) w / iw;
                double hr = (double) h / ih;
                double mf = (hr < wr) ? hr : wr;
                int nw = (int) (mf * iw);
                int nh = (int) (mf * ih);
                int type = (parent.programValues.fadeMode || (image.getTransparency() != Transparency.OPAQUE))
                        ? BufferedImage.TYPE_INT_ARGB
                        : BufferedImage.TYPE_INT_RGB;
                scaledImage = new BufferedImage(nw, nh, type);
                Graphics2D tg2 = scaledImage.createGraphics();
                tg2.drawImage(image, 0, 0, nw, nh, this);
                tg2.dispose();
                // this maintains the image aspect ratio
                // with changes in screen size and shape
                offsetw = (nw > w) ? 0 : (w - nw) / 2;
                offseth = (nh > h) ? 0 : (h - nh) / 2;
            }
        }
        if (image != null && !parent.blankScreen) {
            if (scaledImage != null) {
                if (threadRunning && fadeMode && oldScaledImage != null) {
                    g2.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER, alphaInv));
                    g2.drawImage(oldScaledImage, oldOffsetw, oldOffseth, this);
                    g2.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER, alpha));
                }
                g2.drawImage(scaledImage, offsetw, offseth, this);
            }
            if (testPattern) {
                g2.setColor(Color.yellow);
                g2.drawLine(0, 0, 0, h - 1);
                g2.drawLine(0, h - 1, w - 1, h - 1);
                g2.drawLine(w - 1, h - 1, w - 1, 0);
                g2.drawLine(w - 1, 0, 0, 0);
            }
        }
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        setBackground(java.awt.Color.black);
        setBorder(null);
        setDoubleBuffered(false);
        setFocusable(false);
        addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                formMouseClicked(evt);
            }
        });
        setLayout(new java.awt.GridBagLayout());
    }// </editor-fold>//GEN-END:initComponents

    private void formMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_formMouseClicked
        // TODO add your handling code here:
        parent.handleMouseClick(evt);
    }//GEN-LAST:event_formMouseClicked

    // Variables declaration - do not modify//GEN-BEGIN:variables
    // End of variables declaration//GEN-END:variables
}
