/***************************************************************************
 *   Copyright (C) 2009 by Paul Lutus                                      *
 *   lutusp@arachnoid.com                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
package magiclantern;

import java.awt.*;
import javax.swing.*;
import java.io.*;
import java.util.*;
import java.util.Scanner.*;
import java.awt.event.*;
import javax.imageio.*;
import java.awt.image.*;
import java.net.*;

/**
 *
 * @author lutusp
 */
final public class MagicLantern extends javax.swing.JFrame {

    final String appVersion = "1.1";
    String appPath;
    String fileSep;
    String lineSep;
    String escapedFileSep;
    String configPath;
    String extPath;
    String userDir;
    String userPath;
    String dataPath;
    String defaultFilenameFilter = "(?i).*\\.(png|jpe?g)$";
    String browseContentPath;
    final String appName;
    final String programName;
    Thread outOfMemoryThread;
    Date outOfMemoryFlag = null;
    MyHelpPane helpPane;
    FullScreenFrame fullFrame = null;
    ImagePanel smallImage;
    EditBase editBase;
    ScaledImageGenerator thumbnailGenerator;
    SetupControlPanel setupControlPanel;
    ShowList showList[];
    int editColumns = 4;
    String thumbnailDir = "MagicLanternThumbnails";
    String scaledImageDir = "MagicLanternScaledImages";
    String disabledTitle = "-- Select Show --";
    boolean fullFrameMode = false;
    boolean blankScreen = false;
    boolean lastAutoSlideFlag = false;
    boolean editPreview = false;
    int priorTab = -1;
    InitManager initManager;
    ProgramValues programValues;
    String[] autoShowDurations = {"2", "5", "10", "15", "20", "25", "30"};

    /** Creates new form SlideProjectorFrame */
    public MagicLantern() {
        initComponents();
        appName = getClass().getSimpleName();
        URL url = getClass().getResource(appName + ".class");
        appPath = url.getPath().replaceFirst("(.*?)!.*", "$1");
        appPath = appPath.replaceFirst("file:", "");
        appPath = new File(appPath).getPath();
        programName = appName + " " + appVersion;
        setTitle(programName);
        setIconImage(new ImageIcon(getClass().getResource("icons/" + appName + ".png")).getImage());
        userDir = System.getProperty("user.home");
        lineSep = System.getProperty("line.separator");
        fileSep = System.getProperty("file.separator");
        userPath = userDir + fileSep + "." + appName;
        // show data subdirectory
        dataPath = userPath + fileSep + "data";
        // subdirectory for extension shell scripts
        extPath = userPath + fileSep + "extensions";
        // location of configuration
        configPath = userPath + fileSep + appName + ".ini";
        browseContentPath = userPath + fileSep + "browseContent.txt";
        programValues = new ProgramValues();
        initManager = new InitManager(this, configPath, programValues);
        //now we have restored configuration values
        editColumns = programValues.editColumns;
        autoRepeatCheckBox.setSelected(programValues.autoRepeat);

        CommonCode.testMakeDirs(extPath);
        fullFrame = new FullScreenFrame(this);

        setBounds(programValues.screenBounds);

        smallImage = new ImagePanel(this, "main");
        smallImage.setBorder(null);

        CommonCode.addComponent(smallImage, ImagePanel, 0, 0);

        helpPane = new MyHelpPane(this);

        CommonCode.addComponent(helpPane, helpTab);

        thumbnailGenerator = new ScaledImageGenerator(this);
        CommonCode.addComponent(thumbnailGenerator, setupPanel, 0, 0, 2, 1.0, 1.0);
        setupControlPanel = new SetupControlPanel(this);
        CommonCode.addComponent(setupControlPanel, setupPanel, 0, 1, 2, 1.0, 1.0);
        CommonCode.testMakeDirs(dataPath);
        createShowList(ShowListComboBox, dataPath);
        setScreenState(programValues.screenState);
        editBase = new EditBase(this);
        CommonCode.addComponent(editBase, editTab);
        readShowFile(false);
        startStopAutoShow(false);
        if (programValues.currentTab == 0) {
            requestFocus();
        }
        setupOomMonitor();
    }

    void setupOomMonitor() {
        outOfMemoryThread = new Thread() {

            public void run() {
                outOfMemoryMonitor();
            }
        };
        outOfMemoryThread.start();
    }

    void outOfMemoryMonitor() {
        while (true) {
            //System.out.println("monitoring ...");
            try {
                Date d = new Date();
                // to be valid, the out of memory condition should
                // last longer than two seconds
                boolean problem = (outOfMemoryFlag != null && outOfMemoryFlag.getTime() < d.getTime() - 2000);
                if (problem) {
                    outOfMemoryFlag = null;
                    Runtime.getRuntime().gc();
                    showMemoryWarning();
                }
                Thread.sleep(2000);
            } catch (Exception e) {
                System.out.println("Main:outOfMemoryMonitor(): " + e);
            }
        }
    }

    // an effort to catch out of memory errors
    public BufferedImage readImage(File f) {
        // last-minute sanity check
        if (!f.exists()) {
            return null;
        }
        BufferedImage im = null;
        try {
            Runtime.getRuntime().gc();
            if (outOfMemoryFlag != null) {
                // we're already down the rabbit hole
                im = ImageIO.read(f);
            } else {
                // bracket the critical action with
                // the error flag for later detection
                outOfMemoryFlag = new Date();
                im = ImageIO.read(f);
                outOfMemoryFlag = null;
            }
        } catch (Exception e) {
            System.out.println("main:readImage() : " + e);
        }
        return im;
    }

    public BufferedImage readImage(String path) {
        return readImage(new File(path));
    }

    public void showMemoryWarning() {
        CommonCode.beep();
        String message = programName + " is very low on memory, a problem\n";
        message += "usually caused by large image files. To fix this,\n";
        message += "please read the help section on \"Memory Problems\".";
        JOptionPane.showMessageDialog(this,
                message,
                programName + ": Memory Problem", JOptionPane.WARNING_MESSAGE);
    }

    public void performExtension(String path) {
        File tpath = new File(extPath);
        String list[] = tpath.list();
        if (list != null) {
            for (int i = 0; i < list.length; i++) {
                String com = extPath + fileSep + list[i];
                com = "\"" + com + "\" \"" + path + "\"";
                //System.out.println(com);
                try {
                    Process p = Runtime.getRuntime().exec(com);
                } catch (Exception e) {
                    System.out.println("performExtension: " + e);
                }
            }
        }
    }

    public void setStatus(String s, boolean bell, int tabIndex) {
        if (getTab() == tabIndex) {
            statusLabel.setText(s);
            if (bell) {
                CommonCode.beep();
            }
        }
    }

    String[] readFileList(String dPath) {
        File tpath = new File(dPath);
        MyFileNameFilter fnf = new MyFileNameFilter(".*\\.txt$");
        String list[] = tpath.list(fnf);
        if (list != null) {
            for (int i = 0; i < list.length; i++) {
                list[i] = list[i].replaceFirst("(?i)(.*).txt", "$1");
            }
            Arrays.sort(list, new StringComparatorNoCase());
        }
        return list;
    }

    void fillComboBox(JComboBox box, String[] data) {
        CommonCode.populateComboBox(box, data, 0);
        box.insertItemAt(disabledTitle, 0);
    }

    public void buildShowList(JComboBox box, String dPath) {
        String list[] = readFileList(dPath);
        fillComboBox(box, list);
    }

    public void updateShowList() {
        String currentSelection = (String) ShowListComboBox.getSelectedItem();
        buildShowList(ShowListComboBox, dataPath);
        ShowListComboBox.setSelectedItem(currentSelection);

    }

    void createShowList(JComboBox box, String dPath) {
        buildShowList(box, dPath);
        programValues.showIndex = CommonCode.min(programValues.showIndex, box.getItemCount() - 1);
        programValues.showIndex = CommonCode.max(programValues.showIndex, 0);
        if (programValues.showIndex < box.getItemCount()) {
            box.setSelectedIndex(programValues.showIndex);
        }
    }

    void comboActionPerformed(ActionEvent evt) {
        if ((evt.getModifiers() & ActionEvent.MOUSE_EVENT_MASK) != 0) {
            readShowFile(true);
        }
    }

    void readShowFile(boolean newShow) {
        programValues.showIndex = ShowListComboBox.getSelectedIndex();
        String fn = (String) ShowListComboBox.getSelectedItem();
        String data = CommonCode.readTextFile(dataPath + fileSep + fn + ".txt", lineSep);
        // force zero length array as required
        String array[] = (data == null || data.length() == 0) ? new String[]{} : data.split(lineSep);
        // create two show lists so we can switch back and forth
        // between a map and a slide
        showList = new ShowList[]{new ShowList(array, this), new ShowList(array, this)};
        if (!newShow) {
            showList[0].setIndex(programValues.show0Index);
            showList[1].setIndex(programValues.show1Index);
        }
        if (programValues.currentTab == 0) {
            setStatus("Read " + array.length + " images.", false, 0);
            // must run later so image pane has nonzero size
            SwingUtilities.invokeLater(
                    new Runnable() {

                        public void run() {
                            showSlide(0, false);
                        }
                    });
        }
    }

    void swapShows() {
        programValues.showListIndex = (++programValues.showListIndex) % 2;
        showSlide(0, false);
    }

    boolean showSlide(int n, boolean force) {
        boolean result = true;
        blankScreen = false;
        if (!editPreview) {
            //System.out.println("show slide");
            // defend against nonsense values
            programValues.showListIndex = (programValues.showListIndex < 0) ? -programValues.showListIndex : programValues.showListIndex;
            programValues.showListIndex %= 2;
            if (showList != null) {
                if (fullFrameMode) {
                    fullFrame.image().newImage(n, showList[programValues.showListIndex].getName(n), force);
                } else {
                    smallImage.newImage(n, showList[programValues.showListIndex].getName(n), force);
                }
            } else {
                CommonCode.beep();
            }
        }
        Runtime.getRuntime().gc();
        if (fullFrameMode) {
            fullFrame.requestFocus();
        } else {
            requestFocus();
        }
        return result;
    }

    void manageFullScreen(boolean full) {
        GraphicsEnvironment graphicsEnvironment = GraphicsEnvironment.getLocalGraphicsEnvironment();
        GraphicsDevice graphicsDevice = graphicsEnvironment.getDefaultScreenDevice();
        try {

            if (full) {
                fullFrameMode = true;
                //fullFrame.setVisible(true);
                graphicsDevice.setFullScreenWindow(fullFrame);
                smallImage.clearImage();
            } else {
                graphicsDevice.setFullScreenWindow(null);
                fullFrame.setVisible(false);
                fullFrame.image().clearImage();
                fullFrameMode = false;
            }
            Runtime.getRuntime().gc();
            if (showList != null) {
                java.awt.EventQueue.invokeLater(new Runnable() {

                    public void run() {
                        showSlide(0, true); // update either display
                    }
                });
            }
        } catch (Exception e) {
            System.out.println("manageFullScreen: " + e);
        }
    }

    void toggle_testPattern() {
        if (fullFrameMode) {
            fullFrame.image().toggleTestPattern();
        } else {
            smallImage.toggleTestPattern();
        }
    }

    void selectTab(int n) {
        tabbedPane.setSelectedIndex(n);
        programValues.currentTab = n;
        if (n == 0) {
            //requestFocus();
        }
    }

    public int getTab() {
        return tabbedPane.getSelectedIndex();
    }

    void getCurrentTab() {
        if (priorTab == 1) { ///editing
            editBase.editPanel.saveFileOnExit();
        }
        programValues.currentTab = tabbedPane.getSelectedIndex();
        // if setup tab
        if (programValues.currentTab == 2) {
            setupControlPanel.updateMemStats();
        }
        int screenHash[] = {3, 1, 2, 0};
        setScreenState(screenHash[programValues.currentTab]);
        if (programValues.currentTab == 0) {
            //requestFocus();
        }
        priorTab = programValues.currentTab;
    }

    void editPreview(File file) {
        setScreenState(3);
        editPreview = true;
        smallImage.newImage(0, file.getPath(), true);
        String htmlMessage = "<html>Previewing " + file.getName() + "<br/>Click image to return to editor.</html>";
        String plainMessage = "Previewing " + file.getName() + " -- click image to return to editor.";
        setStatus(plainMessage, false, 0);
        smallImage.setToolTipText(htmlMessage);
    }

    void handleMouseClick(MouseEvent evt) {
        if (editPreview) {
            smallImage.setToolTipText("");
            setStatus("Status", false, 0);
            setScreenState(1);
            editPreview = false;
        }
    }

    void toggleBlankScreen() {
        blankScreen = !blankScreen;
        repaint();
    //showSlide(0, true);
    //CommonCode.beep();
    }

    // 0 = normal, 1 = max, 2 = fullscreen
    void setScreenState(int n) {
        programValues.screenState = n;
        switch (n) {
            case 5: // view fullscreen
                manageFullScreen(true);
                selectTab(0);
                break;
            case 4: // view maximized
                manageFullScreen(false);
                setExtendedState(getExtendedState() | JFrame.MAXIMIZED_BOTH);
                selectTab(0);
                break;
            case 3: // view small
                manageFullScreen(false);
                setExtendedState(getExtendedState() & ~JFrame.MAXIMIZED_BOTH);
                selectTab(0);
                break;
            case 2: // setup
                autoShowOff();
                manageFullScreen(false);
                setExtendedState(getExtendedState() & ~JFrame.MAXIMIZED_BOTH);
                selectTab(2);
                break;
            case 1: // edit
                autoShowOff();
                manageFullScreen(false);
                setExtendedState(getExtendedState() & ~JFrame.MAXIMIZED_BOTH);
                selectTab(1);
                break;
            case 0: // help
                autoShowOff();
                manageFullScreen(false);
                setExtendedState(getExtendedState() & ~JFrame.MAXIMIZED_BOTH);
                selectTab(3);
                helpPane.setFocus();
                break;
        }
    }


    // Typical slide controllers are meant to work with Microsoft PowerPoint
    // so they roughly support the same characters:
    // PgUp = prior slide, PgDn = next slide
    // Esc/F5 = End show (I use it to toggle map slide)
    // Spacebar/N/Right Arraw/Down Arrow/PgDn = next slide
    // Backspace/P/Left Arrow/Up Arrow/Page Up = prior slide
    // B/Period = toggle blank screen
    void handleKeyPressed(KeyEvent evt) {
        String code = KeyEvent.getKeyText(evt.getKeyCode());
        //System.out.println("Key Code: " + code);
        evt.consume();
        if (code.equals("Period") || code.equals("B")) {
            toggleBlankScreen();
        } else if (code.equals("F12")) {
            close(); // exit the program
        } else if (code.equals("F8")) {
            startStopAutoShow(true);
        } else if (code.equals("F7")) {
            setScreenState(2); // configure
        } else if (code.equals("F6")) {
            toggle_testPattern();
        } else if (code.equals("F5")) {
            if (programValues.screenState == 5) {
                swapShows();
            } else {
                setScreenState(5); // full-screen display
            }
        } else if (code.equals("F4")) {
            setScreenState(4); // maximized display
        } else if (code.equals("F3")) {
            setScreenState(3); // small display
        } else if (code.equals("F2")) {
            setScreenState(1); // edit
        } else if (code.equals("F1")) {
            setScreenState(0); // help
        } else if (code.equals("Right") || code.equals("Down") || code.equals("Space") || code.equals("Page Down") || code.equals("N")) {
            showSlide(1, false);
        } else if (code.equals("Backspace") || code.equals("P") || code.equals("Left") || code.equals("Up") || code.equals("Page Up")) {
            showSlide(-1, false);
        } else if (code.equals("Home")) {
            showSlide(-1000000, false);
        } else if (code.equals("End")) {
            showSlide(1000000, false);
        } else if (code.equals("Escape")) {
            swapShows();
        }

    }

    void startStopAutoShow(boolean toggle) {
        if (toggle) {
            programValues.autoShow = !programValues.autoShow;
        }
        AutoShowButton.setSelected(programValues.autoShow);
        if (programValues.autoShow) {
            Thread t = new Thread() {

                @Override
                public void run() {
                    performAutoShow();
                }
            };
            t.start();
        }
    }

    void autoShowOff() {
        AutoShowButton.setSelected(false);
        programValues.autoShow = false;
    }

    void performAutoShow() {
        try {
            // show first slide
            showSlide(-1000000, false);
            int duration = programValues.autoShowSlideDurationSeconds;
            while (programValues.autoShow) {
                tabbedPane.setSelectedIndex(0);
                // allow interruption at 100 msec intervals
                for (int i = 0; i < 10 && programValues.autoShow; i++) {
                    Thread.sleep(duration * 100);
                }
                if (!programValues.autoShow) {
                    return;
                }

                if (lastAutoSlideFlag) {
                    if (programValues.autoRepeat) {
                        showSlide(-1000000, false);
                    } else {
                        autoShowOff();
                        return;
                    }
                } else {
                    showSlide(1, false); // next slide
                }
            }
        } catch (Exception e) {
            System.out.println("performAutoShow: " + e);
        }
    }

    void updateThumbnailList(JList list) {
        MyDefaultListModel model = (MyDefaultListModel) list.getModel();
        for (int i = 0; i < model.size(); i++) {
            EditCellDatum ecd = (EditCellDatum) model.get(i);
            ecd.testLoadThumbnail();
            model.contentsChanged(list, i, i);
        }
    }

    void updateThumbnails() {
        updateThumbnailList(editBase.editPanel.getList());
        updateThumbnailList(editBase.browsePanel.getList());
    }

    void setAutoRepeat() {
        programValues.autoRepeat = autoRepeatCheckBox.isSelected();
    }

    void close() {
        if (editBase.editPanel.saveFileOnExit()) {
            // try to mute sound effects
            performExtension("");
            editBase.browsePanel.saveBrowseFile();
            if (showList != null) {
                programValues.show0Index = showList[0].getIndex();
                programValues.show1Index = showList[1].getIndex();
            }
            programValues.autoRepeat = autoRepeatCheckBox.isSelected();
            programValues.dividerLocation = editBase.getDividerLocation();
            programValues.editColumns = editColumns;
            programValues.screenBounds = getBounds();
            initManager.writeConfig();
            setVisible(false);
            dispose();
            System.exit(0);
        }
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        StatusPanel = new javax.swing.JPanel();
        statusLabel = new javax.swing.JLabel();
        tabbedPane = new javax.swing.JTabbedPane();
        ImageTab = new javax.swing.JPanel();
        TopPanel = new javax.swing.JPanel();
        ShowListComboBox = new javax.swing.JComboBox();
        autoRepeatCheckBox = new javax.swing.JCheckBox();
        AutoShowButton = new javax.swing.JToggleButton();
        toBeginButton = new javax.swing.JButton();
        priorButton = new javax.swing.JButton();
        nextButton = new javax.swing.JButton();
        toEndButton = new javax.swing.JButton();
        swapButton = new javax.swing.JButton();
        blankButton = new javax.swing.JButton();
        quitButton = new javax.swing.JButton();
        ImagePanel = new javax.swing.JPanel();
        editTab = new javax.swing.JPanel();
        setupPanel = new javax.swing.JPanel();
        helpTab = new javax.swing.JPanel();

        setDefaultCloseOperation(javax.swing.WindowConstants.DO_NOTHING_ON_CLOSE);
        setMinimumSize(new java.awt.Dimension(200, 100));
        setName("MainFrame"); // NOI18N
        addWindowListener(new java.awt.event.WindowAdapter() {
            public void windowClosing(java.awt.event.WindowEvent evt) {
                formWindowClosing(evt);
            }
        });
        addKeyListener(new java.awt.event.KeyAdapter() {
            public void keyPressed(java.awt.event.KeyEvent evt) {
                formKeyPressed(evt);
            }
        });
        getContentPane().setLayout(new java.awt.GridBagLayout());

        StatusPanel.setFocusable(false);
        StatusPanel.setLayout(new java.awt.GridBagLayout());

        statusLabel.setText("Status");
        statusLabel.setFocusable(false);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 2, 2, 2);
        StatusPanel.add(statusLabel, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        getContentPane().add(StatusPanel, gridBagConstraints);

        tabbedPane.setTabPlacement(javax.swing.JTabbedPane.BOTTOM);
        tabbedPane.setFocusable(false);
        tabbedPane.setMinimumSize(new java.awt.Dimension(100, 100));
        tabbedPane.setPreferredSize(new java.awt.Dimension(600, 400));
        tabbedPane.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                tabbedPaneMouseClicked(evt);
            }
        });

        ImageTab.setBorder(null);
        ImageTab.setFocusable(false);
        ImageTab.setMinimumSize(new java.awt.Dimension(200, 100));
        ImageTab.setPreferredSize(new java.awt.Dimension(400, 300));
        ImageTab.setLayout(new java.awt.GridBagLayout());

        TopPanel.setFocusable(false);

        ShowListComboBox.setToolTipText("Select a slide show");
        ShowListComboBox.setFocusable(false);
        ShowListComboBox.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                ShowListComboBoxActionPerformed(evt);
            }
        });
        TopPanel.add(ShowListComboBox);

        autoRepeatCheckBox.setText("Repeat");
        autoRepeatCheckBox.setToolTipText("This option causes the auto show to repeat");
        autoRepeatCheckBox.setFocusable(false);
        autoRepeatCheckBox.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                autoRepeatCheckBoxMouseClicked(evt);
            }
        });
        TopPanel.add(autoRepeatCheckBox);

        AutoShowButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/magiclantern/icons/network-offline.png"))); // NOI18N
        AutoShowButton.setText("Auto");
        AutoShowButton.setToolTipText("Start/Stop Auto Show (keyboard F8)");
        AutoShowButton.setDisabledIcon(new javax.swing.ImageIcon(getClass().getResource("/magiclantern/icons/network-offline.png"))); // NOI18N
        AutoShowButton.setFocusable(false);
        AutoShowButton.setRolloverEnabled(false);
        AutoShowButton.setSelectedIcon(new javax.swing.ImageIcon(getClass().getResource("/magiclantern/icons/network-receive.png"))); // NOI18N
        AutoShowButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                AutoShowButtonMouseClicked(evt);
            }
        });
        TopPanel.add(AutoShowButton);

        toBeginButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/magiclantern/icons/go-first.png"))); // NOI18N
        toBeginButton.setToolTipText("First Slide (keyboard Home)");
        toBeginButton.setFocusable(false);
        toBeginButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                toBeginButtonMouseClicked(evt);
            }
        });
        TopPanel.add(toBeginButton);

        priorButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/magiclantern/icons/go-previous.png"))); // NOI18N
        priorButton.setToolTipText("Prior Slide (keyboard PgUp or Left Arrow)");
        priorButton.setFocusable(false);
        priorButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                priorButtonMouseClicked(evt);
            }
        });
        TopPanel.add(priorButton);

        nextButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/magiclantern/icons/go-next.png"))); // NOI18N
        nextButton.setToolTipText("Next Slide (keyboard PgDn or Right Arrow)");
        nextButton.setFocusable(false);
        nextButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                nextButtonMouseClicked(evt);
            }
        });
        TopPanel.add(nextButton);

        toEndButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/magiclantern/icons/go-last.png"))); // NOI18N
        toEndButton.setToolTipText("Last Slide (keyboard End)");
        toEndButton.setFocusable(false);
        toEndButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                toEndButtonMouseClicked(evt);
            }
        });
        TopPanel.add(toEndButton);

        swapButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/magiclantern/icons/view-refresh.png"))); // NOI18N
        swapButton.setToolTipText("Swap Image/Map (keyboard Escape or F5)");
        swapButton.setFocusable(false);
        swapButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                swapButtonMouseClicked(evt);
            }
        });
        TopPanel.add(swapButton);

        blankButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/magiclantern/icons/system-lock-screen.png"))); // NOI18N
        blankButton.setToolTipText("Blank Screen (keyboard \".\")");
        blankButton.setFocusable(false);
        blankButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                blankButtonMouseClicked(evt);
            }
        });
        TopPanel.add(blankButton);

        quitButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/magiclantern/icons/application-exit.png"))); // NOI18N
        quitButton.setToolTipText("Exit (F12)");
        quitButton.setFocusable(false);
        quitButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                quitButtonMouseClicked(evt);
            }
        });
        TopPanel.add(quitButton);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTH;
        gridBagConstraints.weightx = 1.0;
        ImageTab.add(TopPanel, gridBagConstraints);

        ImagePanel.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                ImagePanelMouseClicked(evt);
            }
        });
        ImagePanel.setLayout(new java.awt.GridBagLayout());
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        ImageTab.add(ImagePanel, gridBagConstraints);

        tabbedPane.addTab("Display", new javax.swing.ImageIcon(getClass().getResource("/magiclantern/icons/applications-multimedia.png")), ImageTab); // NOI18N

        editTab.setFocusable(false);
        editTab.setLayout(new java.awt.GridBagLayout());
        tabbedPane.addTab("Create/Edit", new javax.swing.ImageIcon(getClass().getResource("/magiclantern/icons/applications-graphics.png")), editTab); // NOI18N

        setupPanel.setLayout(new java.awt.GridBagLayout());
        tabbedPane.addTab("Configure", new javax.swing.ImageIcon(getClass().getResource("/magiclantern/icons/applications-accessories.png")), setupPanel); // NOI18N

        helpTab.setFocusable(false);
        helpTab.setLayout(new java.awt.GridBagLayout());
        tabbedPane.addTab("Help", new javax.swing.ImageIcon(getClass().getResource("/magiclantern/icons/system-help.png")), helpTab); // NOI18N

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        getContentPane().add(tabbedPane, gridBagConstraints);

        pack();
    }// </editor-fold>//GEN-END:initComponents

    private void priorButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_priorButtonMouseClicked
        // TODO add your handling code here:
        showSlide(-1, false);
    }//GEN-LAST:event_priorButtonMouseClicked

    private void nextButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_nextButtonMouseClicked
        // TODO add your handling code here:
        showSlide(1, false);
    }//GEN-LAST:event_nextButtonMouseClicked

    private void toBeginButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_toBeginButtonMouseClicked
        // TODO add your handling code here:
        showSlide(-1000000, false);
    }//GEN-LAST:event_toBeginButtonMouseClicked

    private void toEndButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_toEndButtonMouseClicked
        // TODO add your handling code here:
        showSlide(1000000, false);
    }//GEN-LAST:event_toEndButtonMouseClicked

    private void swapButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_swapButtonMouseClicked
        // TODO add your handling code here:
        swapShows();
    }//GEN-LAST:event_swapButtonMouseClicked

    private void formKeyPressed(java.awt.event.KeyEvent evt) {//GEN-FIRST:event_formKeyPressed
        // TODO add your handling code here:
        handleKeyPressed(evt);
    }//GEN-LAST:event_formKeyPressed

    private void formWindowClosing(java.awt.event.WindowEvent evt) {//GEN-FIRST:event_formWindowClosing
        // TODO add your handling code here:
        close();
    }//GEN-LAST:event_formWindowClosing

    private void quitButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_quitButtonMouseClicked
        // TODO add your handling code here:
        close();
    }//GEN-LAST:event_quitButtonMouseClicked

    private void tabbedPaneMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_tabbedPaneMouseClicked
        // TODO add your handling code here:
        getCurrentTab();
    }//GEN-LAST:event_tabbedPaneMouseClicked

    private void blankButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_blankButtonMouseClicked
        // TODO add your handling code here:
        toggleBlankScreen();
    }//GEN-LAST:event_blankButtonMouseClicked

    private void AutoShowButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_AutoShowButtonMouseClicked
        // TODO add your handling code here:
        startStopAutoShow(true);
}//GEN-LAST:event_AutoShowButtonMouseClicked

    private void autoRepeatCheckBoxMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_autoRepeatCheckBoxMouseClicked
        // TODO add your handling code here:
        setAutoRepeat();
    }//GEN-LAST:event_autoRepeatCheckBoxMouseClicked

    private void ImagePanelMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_ImagePanelMouseClicked
        // TODO add your handling code here:
        handleMouseClick(evt);
    }//GEN-LAST:event_ImagePanelMouseClicked

    private void ShowListComboBoxActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_ShowListComboBoxActionPerformed
        // TODO add your handling code here:
        comboActionPerformed(evt);
    }//GEN-LAST:event_ShowListComboBoxActionPerformed

    /**
     * @param args the command line arguments
     */
    public static void main(String args[]) {

        try {
            // Default to system-specific L&F
            UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
        } catch (Exception e) {
            // handle exception
        }
        java.awt.EventQueue.invokeLater(new Runnable() {

            public void run() {
                try {
                    new MagicLantern().setVisible(true);
                } catch (Exception e) {
                    System.out.println("Main (last resort exception trap): " + e);
                }
            }
        });
    }

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JToggleButton AutoShowButton;
    private javax.swing.JPanel ImagePanel;
    private javax.swing.JPanel ImageTab;
    private javax.swing.JComboBox ShowListComboBox;
    private javax.swing.JPanel StatusPanel;
    private javax.swing.JPanel TopPanel;
    private javax.swing.JCheckBox autoRepeatCheckBox;
    private javax.swing.JButton blankButton;
    private javax.swing.JPanel editTab;
    private javax.swing.JPanel helpTab;
    private javax.swing.JButton nextButton;
    private javax.swing.JButton priorButton;
    private javax.swing.JButton quitButton;
    private javax.swing.JPanel setupPanel;
    private javax.swing.JLabel statusLabel;
    private javax.swing.JButton swapButton;
    private javax.swing.JTabbedPane tabbedPane;
    private javax.swing.JButton toBeginButton;
    private javax.swing.JButton toEndButton;
    // End of variables declaration//GEN-END:variables
}
