/***************************************************************************
 *   Copyright (C) 2009 by Paul Lutus                                      *
 *   lutusp@arachnoid.com                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
package magiclantern;

import javax.swing.*;
import java.io.*;
import java.util.*;
import java.awt.*;
import java.awt.event.*;
import javax.imageio.*;
import java.awt.image.*;

/**
 *
 * @author lutusp
 */
final public class ScaledImageGenerator extends javax.swing.JPanel {

    MagicLantern parent;
    boolean threadRunning = false;
    int generatedImageCount = 0;
    String baseDir;
    String thumbPrefix = "thumb_";
    String scaledPrefix = "scaled_";
    String[] qualityLabels = {"Good", "Better", "Best"};
    Object[] rederingHints = {
        RenderingHints.VALUE_INTERPOLATION_NEAREST_NEIGHBOR,
        RenderingHints.VALUE_INTERPOLATION_BILINEAR,
        RenderingHints.VALUE_INTERPOLATION_BICUBIC
    };
    Object renderHint;

    /** Creates new form ThumbnailGenerator */
    public ScaledImageGenerator(MagicLantern p) {
        parent = p;
        initComponents();
        createHelpText();
        setScaleMode();
        setupQualityCombo();
        setPickPathButtonTip();
    }

    void setupQualityCombo() {
        CommonCode.populateComboBox(qualityComboBox, qualityLabels, parent.programValues.scaleQualityIndex);
    }

    void actionQualityCombo(ActionEvent evt) {
        if (!threadRunning) {
            if ((evt.getModifiers() & ActionEvent.MOUSE_EVENT_MASK) != 0) {
                parent.programValues.scaleQualityIndex = qualityComboBox.getSelectedIndex();
            }
        }
    }

    void setScaleMode() {
        if (!threadRunning) {
            // have to do it this way
            if (parent.programValues.thumbnailMode) {
                thumbnailRadioButton.setSelected(true);
            } else {
                presentationRadioButton.setSelected(true);
            }
            qualityComboBox.setEnabled(!parent.programValues.thumbnailMode);
            setCurrentScaleEntry();
        }
    }

    void getScaleMode() {
        if (!threadRunning) {
            parent.programValues.thumbnailMode = thumbnailRadioButton.isSelected();
            setScaleMode();
        }
    }

    String fetchStorageDirectory() {
        return ((parent.programValues.thumbnailMode) ? parent.thumbnailDir : parent.scaledImageDir);
    }

    int fetchImageScale() {
        return ((parent.programValues.thumbnailMode) ? parent.programValues.thumbnailImageSize : parent.programValues.scaledImageSize);
    }

    String fetchImageNamePrefix() {
        return ((parent.programValues.thumbnailMode) ? thumbPrefix : scaledPrefix);
    }

    void setCurrentScaleEntry() {
        scaledSizeTextField.setText("" + fetchImageScale());
        controlPanel.revalidate();
    }

    void createHelpText() {
        String text = "This generator creates scaled images for two purposes:\n\n";
        text += "   1. In thumbnail mode, the generator creates tiny versions of full-sized graphic images (called \"thumbnails\") needed by the MagicLantern editor.\n\n";
        text += "   2. In scaled mode, the generator creates scaled-down versions of large images for presentation purposes.\n\n";
        text += "Creating scaled images can take a lot of time, so it may be better to run this generator during idle time.\n\n";
        text += "As images are added to your collection, you may want to visit this dialog to create any missing scaled versions -- ";
        text += "the generator will only create scaled images that are missing or that are the wrong size.\n\n";
        text += "This dialog may also be used to change the size all scaled images or thumbnails at once, although because of the time and processing power required, ";
        text += "this is an activity best reserved for times when your computer isn't being stressed (this generator runs as a background task).\n\n";
        text += "To use the generator:\n\n";
        text += "   1. Choose a mode (thumbnail or scaled)\n";
        text += "   2. For the scaled image mode, choose a quality level (higher quality takes more time)\n";
        text += "   3. Choose a scale size or (preferred) accept the default\n";
        text += "   4. Choose a directory to scan (all subdirectories will be included)\n";
        text += "   5. Press \"Start\" to begin the process.\n";
        scaleHelpTextArea.setText(text);
        scaleHelpTextArea.setSelectionStart(0);
        scaleHelpTextArea.setSelectionEnd(0);
    }

    void setPickPathButtonTip() {
        String s = "Select a base directory for this activity";
        String p = parent.programValues.scaledImageGeneratorDirectory;
        if (p.length() > 0) {
            s = "<html>Choose a directory<br/>(now set to: " + p + ")</html>";
        }
        pickPathButton.setToolTipText(s);
    }

    void chooseDirectory() {
        if (!threadRunning) {
            JFileChooser chooser = new JFileChooser();
            boolean changed = CommonCode.changeLookAndFeel(chooser);
            chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
            chooser.setDialogTitle(parent.programName + ": Choose Scaled Image Base Directory");
            chooser.setApproveButtonText("Accept");
            chooser.setCurrentDirectory(new File(parent.programValues.scaledImageGeneratorDirectory));
            int returnVal = chooser.showOpenDialog(parent);
            if (changed) {
                CommonCode.restoreSystemLookAndFeel();
            }
            if (returnVal == JFileChooser.APPROVE_OPTION) {
                try {
                    parent.programValues.scaledImageGeneratorDirectory = chooser.getSelectedFile().getCanonicalPath();
                    setPickPathButtonTip();
                } catch (Exception e) {
                    System.out.println(e);
                }
            }
        }
    }

    void launchGenerateThread() {
        if (!threadRunning) {
            baseDir = CommonCode.strip(parent.programValues.scaledImageGeneratorDirectory);
            if (baseDir.length() == 0) {
                CommonCode.beep();
                JOptionPane.showMessageDialog(parent, "Please choose a path for image processing", parent.programName + ": No Path Chosen", JOptionPane.WARNING_MESSAGE);
            } else {
                parent.programValues.scaledImageGeneratorDirectory = baseDir;
                renderHint = rederingHints[parent.programValues.scaleQualityIndex];
                updateScaleEntry();
                generatedImageCount = 0;
                Thread t = new Thread() {

                    @Override
                    public void run() {
                        generateThread();
                    }
                };
                t.start();
            }
        }
    }

    void generateThread() {
        threadRunning = true;
        String type = (parent.programValues.thumbnailMode) ? "thumbnails" : "scaled images";
        Vector<File> fileList = new Vector<File>();
        fileList = scanDir(fileList, new File(baseDir));
        Collections.sort(fileList, new FileComparatorNoCase());
        int top = fileList.size();
        int scale = fetchImageScale();
        String namePrefix = fetchImageNamePrefix();
        String storageDirectory = fetchStorageDirectory();
        theProgressBar.setStringPainted(true);
        addProgressText(scaleHelpTextArea, "\nGenerating " + type + ":\n");
        for (int i = 0; i < fileList.size() && threadRunning; i++) {
            int p = (100 * i) / top;
            theProgressBar.setValue(p);
            theProgressBar.setString(p + "%");
            generateScaledImageTest(fileList.get(i), scale, namePrefix, storageDirectory);
        }
        if (!threadRunning) {
            addProgressText(scaleHelpTextArea, "Operation canceled.");
        } else {
            addProgressText(scaleHelpTextArea, "Scanned " + fileList.size() + " graphic files, created " + generatedImageCount + " " + type + ".");
        }
        if (parent.programValues.thumbnailMode) {
            parent.updateThumbnails();
        }
        theProgressBar.setValue(0);
        theProgressBar.setString("Done.");
        threadRunning = false;
    }

    Vector<File> scanDir(Vector<File> vec, File f) {
        File[] list = f.listFiles();
        String name;
        for (int i = 0; i < list.length; i++) {
            //System.out.println(list[i]);
            name = list[i].getName();
            if (list[i].isDirectory()) {
                // avoid recursing into thumbnail directories
                // or other scaled directories if not in thumbnail mode
                if (!((name.equals(parent.thumbnailDir) || (!parent.programValues.thumbnailMode && name.matches(parent.scaledImageDir + ".*"))))) {
                    vec = scanDir(vec, list[i]);
                }
            } else { // test file for inclusion
                // avoid existing scaled images
                if (name.matches(parent.defaultFilenameFilter) && !name.matches(thumbPrefix + ".*")) {
                    vec.add(list[i]);
                }
            }
        }
        return vec;
    }

    void addProgressText(JTextArea control, String s) {
        control.append(s);
        control.setCaretPosition(control.getDocument().getLength() - 1);
    }

    void generateScaledImageTest(File originalImageFile, int scale, String namePref, String storageDir) {
        String basePath = originalImageFile.getParent();
        String fileName = originalImageFile.getName();
        String scaledImageDir = basePath + parent.fileSep + storageDir;
        File ftp = new File(scaledImageDir);
        // create scaled-image subdirectory
        if (!ftp.exists()) {
            ftp.mkdirs(); 
        }
        // create full path to scaled file
        String scaledImagePath = scaledImageDir + parent.fileSep + namePref + fileName;
        try {
            File scaledImageFile = new File(scaledImagePath);
            // test 1: if no scaled image, create it
            if (!scaledImageFile.exists()) {
                createWriteScaledImage(fileName,originalImageFile, scaledImageFile, scale);
            } else { // test 2: if scaled image size is wrong, recreate
                Image im = parent.readImage(scaledImageFile);
                int w = im.getWidth(this);
                int h = im.getHeight(this);
                if (!((w == scale && h <= scale) || (h == scale && w <= scale))) {
                    createWriteScaledImage(fileName,originalImageFile, scaledImageFile, scale);
                } else { // test 3: if image lastModified time is newer than scaled, recreate
                    if (originalImageFile.lastModified() > scaledImageFile.lastModified()) {
                        createWriteScaledImage(fileName,originalImageFile, scaledImageFile, scale);
                    }
                }
            }
        } catch (Exception e) {
            System.out.println("generateScaledImageTest: " + e);
        }
    }

    void createWriteScaledImage(String fileName,File origImageFile, File scaledImageFile, int scale) {
        generatedImageCount++;
        addProgressText(scaleHelpTextArea, "Creating " + scaledImageFile.getPath() + "\n");
        try {
            Runtime.getRuntime().gc();
            //String fpath = path + parent.fileSep + name;
            BufferedImage tmp = parent.readImage(origImageFile);
            if (tmp != null) {
                BufferedImage im = generateScaledImage(tmp, scale);
                // get graphic identifier suffix
                String suffix = fileName.replaceFirst(".*\\.(.*)", "$1");
                ImageIO.write(im, suffix, scaledImageFile);
            }
        } catch (Exception e) {
            System.out.println("createWriteScaledImage error: " + e);
        }
    }

    BufferedImage generateScaledImage(BufferedImage im, int scale) {
        int h = im.getHeight(null);
        int w = im.getWidth(null);
        // fit randomly sized graphics into a square
        // having sides of "scale"
        double ratio = ((double) scale / ((w > h) ? w : h));
        int th = (int) ((h * ratio) + 0.5);
        int tw = (int) ((w * ratio) + 0.5);
        int type = (im.getTransparency() == Transparency.OPAQUE) ? BufferedImage.TYPE_INT_RGB : BufferedImage.TYPE_INT_ARGB;
        BufferedImage out = new BufferedImage(tw, th, type);
        Graphics2D g2 = out.createGraphics();
        if (!parent.programValues.thumbnailMode) {
            // create a better quality image for
            // the large scaled-down graphics
            g2.setRenderingHint(RenderingHints.KEY_INTERPOLATION, renderHint);
        }
        g2.drawImage(im, 0, 0, tw, th, parent);
        g2.dispose();
        return out;
    }

    void updateScaleEntry() {
        if (!threadRunning) {
            try {
                int v = Integer.parseInt(scaledSizeTextField.getText());
                if (v > 0) {
                    if (parent.programValues.thumbnailMode) {
                        parent.programValues.thumbnailImageSize = v;
                    } else {
                        parent.programValues.scaledImageSize = v;
                    }
                }
            } catch (Exception e) {
                System.out.println(e);
                setCurrentScaleEntry();
            }
        }
    }

    void actOnKeyTyped(KeyEvent evt) {
        if (evt.getKeyChar() == '\n') {
            updateScaleEntry();
        }
    }

    void cancelGenerator() {
        threadRunning = false;
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        scalebuttonGroup = new javax.swing.ButtonGroup();
        thumbnailPanel = new javax.swing.JPanel();
        jLabel2 = new javax.swing.JLabel();
        jPanel4 = new javax.swing.JPanel();
        jScrollPane2 = new javax.swing.JScrollPane();
        scaleHelpTextArea = new javax.swing.JTextArea();
        controlPanel = new javax.swing.JPanel();
        thumbnailRadioButton = new javax.swing.JRadioButton();
        presentationRadioButton = new javax.swing.JRadioButton();
        qualityComboBox = new javax.swing.JComboBox();
        jLabel3 = new javax.swing.JLabel();
        scaledSizeTextField = new javax.swing.JTextField();
        pickPathButton = new javax.swing.JButton();
        generateImagesButton = new javax.swing.JButton();
        cancelButton = new javax.swing.JButton();
        jPanel2 = new javax.swing.JPanel();
        theProgressBar = new javax.swing.JProgressBar();

        setMinimumSize(new java.awt.Dimension(541, 200));
        setPreferredSize(new java.awt.Dimension(541, 200));
        setLayout(new java.awt.GridBagLayout());

        thumbnailPanel.setBorder(javax.swing.BorderFactory.createLineBorder(new java.awt.Color(0, 0, 0)));
        thumbnailPanel.setLayout(new java.awt.GridBagLayout());

        jLabel2.setText("<html><b>Scaled Image Generator</b></html>");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        thumbnailPanel.add(jLabel2, gridBagConstraints);

        jPanel4.setLayout(new java.awt.GridBagLayout());

        jScrollPane2.setBorder(javax.swing.BorderFactory.createLineBorder(new java.awt.Color(0, 0, 0)));
        jScrollPane2.setHorizontalScrollBarPolicy(javax.swing.ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER);

        scaleHelpTextArea.setColumns(20);
        scaleHelpTextArea.setEditable(false);
        scaleHelpTextArea.setLineWrap(true);
        scaleHelpTextArea.setRows(5);
        scaleHelpTextArea.setWrapStyleWord(true);
        scaleHelpTextArea.setMargin(new java.awt.Insets(4, 4, 4, 4));
        jScrollPane2.setViewportView(scaleHelpTextArea);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 0.25;
        gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
        jPanel4.add(jScrollPane2, gridBagConstraints);

        controlPanel.setBackground(new java.awt.Color(231, 198, 253));
        controlPanel.setBorder(javax.swing.BorderFactory.createLineBorder(new java.awt.Color(0, 0, 0)));

        thumbnailRadioButton.setBackground(new java.awt.Color(231, 198, 253));
        scalebuttonGroup.add(thumbnailRadioButton);
        thumbnailRadioButton.setSelected(true);
        thumbnailRadioButton.setText("Thumb");
        thumbnailRadioButton.setToolTipText("Create tiny images (\"thumbnails\") for use with the editor");
        thumbnailRadioButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                thumbnailRadioButtonMouseClicked(evt);
            }
        });
        controlPanel.add(thumbnailRadioButton);

        presentationRadioButton.setBackground(new java.awt.Color(231, 198, 253));
        scalebuttonGroup.add(presentationRadioButton);
        presentationRadioButton.setText("Scaled");
        presentationRadioButton.setToolTipText("Create scaled-down images for use in slide shows");
        presentationRadioButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                presentationRadioButtonMouseClicked(evt);
            }
        });
        controlPanel.add(presentationRadioButton);

        qualityComboBox.setModel(new javax.swing.DefaultComboBoxModel(new String[] { "Item 1", "Item 2", "Item 3", "Item 4" }));
        qualityComboBox.setToolTipText("Select the quality level for the scaled mode");
        qualityComboBox.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                qualityComboBoxActionPerformed(evt);
            }
        });
        controlPanel.add(qualityComboBox);

        jLabel3.setText("Scale:");
        jLabel3.setToolTipText("Enter an image scale in pixels");
        controlPanel.add(jLabel3);
        jLabel3.getAccessibleContext().setAccessibleName("");

        scaledSizeTextField.setHorizontalAlignment(javax.swing.JTextField.RIGHT);
        scaledSizeTextField.setToolTipText("<html>Only change this if you understand the implications.<br/>\nRecreating a large set of images can take a long time.</html>");
        scaledSizeTextField.setMinimumSize(new java.awt.Dimension(50, 27));
        scaledSizeTextField.setPreferredSize(new java.awt.Dimension(50, 27));
        scaledSizeTextField.addKeyListener(new java.awt.event.KeyAdapter() {
            public void keyTyped(java.awt.event.KeyEvent evt) {
                scaledSizeTextFieldKeyTyped(evt);
            }
        });
        controlPanel.add(scaledSizeTextField);

        pickPathButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/magiclantern/icons/document-open.png"))); // NOI18N
        pickPathButton.setText("Path");
        pickPathButton.setToolTipText("Select a base directory for this activity");
        pickPathButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                pickPathButtonMouseClicked(evt);
            }
        });
        controlPanel.add(pickPathButton);

        generateImagesButton.setText("Start");
        generateImagesButton.setToolTipText("Start the generator");
        generateImagesButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                generateImagesButtonMouseClicked(evt);
            }
        });
        controlPanel.add(generateImagesButton);

        cancelButton.setText("Stop");
        cancelButton.setToolTipText("Stop/cancel the generation");
        cancelButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                cancelButtonMouseClicked(evt);
            }
        });
        controlPanel.add(cancelButton);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
        jPanel4.add(controlPanel, gridBagConstraints);

        jPanel2.setLayout(new java.awt.GridBagLayout());
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 3, 0, 3);
        jPanel2.add(theProgressBar, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        jPanel4.add(jPanel2, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 16, 0, 0);
        thumbnailPanel.add(jPanel4, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 0.25;
        add(thumbnailPanel, gridBagConstraints);
    }// </editor-fold>//GEN-END:initComponents

    private void pickPathButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_pickPathButtonMouseClicked
        // TODO add your handling code here:
        chooseDirectory();
    }//GEN-LAST:event_pickPathButtonMouseClicked

    private void generateImagesButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_generateImagesButtonMouseClicked
        // TODO add your handling code here:
        launchGenerateThread();
}//GEN-LAST:event_generateImagesButtonMouseClicked

    private void scaledSizeTextFieldKeyTyped(java.awt.event.KeyEvent evt) {//GEN-FIRST:event_scaledSizeTextFieldKeyTyped
        // TODO add your handling code here:
        actOnKeyTyped(evt);
}//GEN-LAST:event_scaledSizeTextFieldKeyTyped

    private void cancelButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_cancelButtonMouseClicked
        // TODO add your handling code here:
        cancelGenerator();
    }//GEN-LAST:event_cancelButtonMouseClicked

    private void thumbnailRadioButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_thumbnailRadioButtonMouseClicked
        // TODO add your handling code here:
        getScaleMode();
    }//GEN-LAST:event_thumbnailRadioButtonMouseClicked

    private void presentationRadioButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_presentationRadioButtonMouseClicked
        // TODO add your handling code here:
        getScaleMode();
    }//GEN-LAST:event_presentationRadioButtonMouseClicked

    private void qualityComboBoxActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_qualityComboBoxActionPerformed
        // TODO add your handling code here:
        actionQualityCombo(evt);
    }//GEN-LAST:event_qualityComboBoxActionPerformed

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton cancelButton;
    private javax.swing.JPanel controlPanel;
    private javax.swing.JButton generateImagesButton;
    private javax.swing.JLabel jLabel2;
    private javax.swing.JLabel jLabel3;
    private javax.swing.JPanel jPanel2;
    private javax.swing.JPanel jPanel4;
    private javax.swing.JScrollPane jScrollPane2;
    private javax.swing.JButton pickPathButton;
    private javax.swing.JRadioButton presentationRadioButton;
    private javax.swing.JComboBox qualityComboBox;
    private javax.swing.JTextArea scaleHelpTextArea;
    private javax.swing.ButtonGroup scalebuttonGroup;
    private javax.swing.JTextField scaledSizeTextField;
    private javax.swing.JProgressBar theProgressBar;
    private javax.swing.JPanel thumbnailPanel;
    private javax.swing.JRadioButton thumbnailRadioButton;
    // End of variables declaration//GEN-END:variables
}
