/***************************************************************************
 *   Copyright (C) 2004 by Paul Lutus                                      *
 *   lutusp@pl-alpha                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#include "iconeditor.h"
#include "slideprojector.h"
#include <qsplitter.h>
#include "myiconview.h"
#include "myiconviewitem.h"
#include <qdragobject.h>
#include <qfiledialog.h>
#include <qcombobox.h>
#include <qmessagebox.h>
#include <qinputdialog.h>
#include <qpushbutton.h>
#include <qwhatsthis.h>

const char *IconEditor::whatsThisText[] = {
"This pane is the location of your slide show. It is the destination for any images you want to include in your show.\n\nTo begin editing, create a new show using \"New/Copy\" above, or load an existing show using the drop-down list above.\n\nTo finish editing and while working, from time to time press \"Save\" above to save your changes. This will prevent loss of your work.\n\nRemember you can undo any actions you have taken by pressing Ctrl+Z, and you can redo those actions by pressing Ctrl+Shift+Z.",

"This pane is a scratchpad area for the temporary storage of images you haven't decided to include in your show. The contents of this pane are not saved.",

"This pane contains a directory listing of images on your computer's filesystem. Remember about this listing that deleting an image from it only removes a name from a list, not a file from a directory. In other words, deletions are harmless and your images will not be lost."
};


IconEditor::IconEditor(QWidget* par, SlideProjector *p,const char* name, WFlags fl)
        : IconEditorMainFrame(par,name,fl) {
    parent = p;
    currentView = NULL;
    //active = false;
    //QGridLayout *leftLayout = new QGridLayout(placeHolderFrame);
    dest = new MyIconView(leftTabWidget,this,parent,whatsThisText[0],NULL,"Dest" );
    QToolTip::add
        (dest, "This panel is the destination for your slide show.");
    scratch = new MyIconView(leftTabWidget,this,parent,whatsThisText[1],NULL,"Scratch" );
    QToolTip::add
        (scratch, "This panel is a temporary area for images.");

    layout10->remove
    (dummyWhatsThisButton);
    delete dummyWhatsThisButton;
    QPushButton *wt = (QPushButton *) QWhatsThis::whatsThisButton(this);
    layout10->addWidget( wt );

    //leftLayout->addWidget(dest,0,0);

    //rightTabWidget->removePage(tab);
    //delete tab;
    rightTabWidget->removePage(tab_2);
    delete tab_2;
    //splitter11->removeChild(rightTabWidget);
    //delete rightTabWidget;
    //rightTabWidget = new QTabWidget();
    leftTabWidget->removePage(tab);
    delete tab;
    leftTabWidget->addTab(dest,"Show Panel");

    updateSaveStatus(dest,false);
    rightTabWidget->addTab(scratch,"Scratchpad");
    updateSaveStatus(scratch,false);
    changed = false;
}

IconEditor::~IconEditor() {
    delete dest;
    delete scratch;
}

/*$SPECIALIZATION$*/
void IconEditor::loadEditDirectory() {
    //printf("addDirectory\n");
    //fflush(stdout);
    QFileDialog fd( this, "Open Image Directory", TRUE );
    fd.setMode( QFileDialog::DirectoryOnly );
    fd.setViewMode( QFileDialog::Detail );
    fd.setCaption("Choose an image directory");
    fd.setSelection(parent->lastImageDirectory);
    if ( fd.exec() == QDialog::Accepted ) {
        string path = fd.selectedFile();
        if(path.length() > 0) {
            //printf("selected file: %s\n",path.c_str());
            //fflush(stdout);
            parent->lastImageDirectory = path;
            MyIconView *view = new MyIconView(this,this,parent,whatsThisText[2],path.c_str(),"Source" );
            rightTabWidget->addTab(view,view->showName);
            rightTabWidget->showPage(view);
            updateSaveStatus(view,false);
        }
    }
}



void IconEditor::loadEditShow() {
    // check for changed here
    if (checkChanged()) {
        string name = loadEditComboBox->currentText();
        if(name != parent->dummyShowName) {
            showPath = parent->progPath + "/" + name + ".txt";
            dest->loadImagesFromListFile(showPath.c_str(),name);
        } else { // dummy name
            dest->clearPanel();
        }
    }
}

void IconEditor::newCom() {
    //printf("new\n");
    //fflush(stdout);
    if (checkChanged()) {
        string name = QInputDialog::getText("Enter new show name","Enter a name for your new show");
        if(name.length() > 0) {
            string path = parent->progPath + "/" + name + ".txt";
            if(fileExists(path)) {
                string message = ((string)"The show \""
                                  + name + "\" already exists.\n"
                                  + "Please choose another name.");
                QMessageBox::QMessageBox::information(this,"Show already exists", message,QMessageBox::Ok);
            } else {
                showPath = path;
                dest->newListFile(showPath,name);
                parent->setupList();
                loadEditComboBox->setCurrentText(name);
            }
        }
    }
}

bool IconEditor::fileExists(string path) {
    ifstream ifs(path.c_str());
    bool r = ifs.is_open();
    ifs.close();
    return r;
}

void IconEditor::saveCom() {
    if(showPath.length() == 0) {
        newCom();
    } else {
        dest->saveImagesToListFile();
        parent->testRefreshAfterSave(dest->showName);
    }
}

bool IconEditor::checkChanged() {
    if(dest->changed) {
        string prompt = (string) "There are changes to slide show \"" + dest->showName + "\". Save this show?";
        int reply = QMessageBox::question(this,"Save changes", prompt
                                          ,QMessageBox::Yes,QMessageBox::No,QMessageBox::Cancel);
        if(reply == QMessageBox::Cancel) {
            return false;
        }
        if(reply == QMessageBox::Yes) {
            dest->saveImagesToListFile();
        }
    }
    return true;
}

void IconEditor::updateSaveStatus(MyIconView *p,bool changed) {
    if (p == dest) {
        if(dest->showName.length() == 0) {
            leftTabWidget->setTabLabel(dest,"Show Panel");
        } else {
            ostringstream ss;
            ss << dest->showName << ": " << dest->count();
            string disp = ss.str();
            int v = dest->count() * parent->editSlideDuration;
            int sec = v % 60;
            v /= 60;
            int min = v % 60;
            int hrs = v / 60;
            char buf[256];
            sprintf(buf," slides, time %02d:%02d:%02d",hrs,min,sec);
            leftTabWidget->setTabLabel(dest,disp+buf);
            saveEditButton->setEnabled(changed);
        }
    } else { // not dest
        ostringstream ss;
        string name = (p == scratch)?"Scratch":p->showName;
        ss << name << " (" << p->count() << ")";
        rightTabWidget->setTabLabel(p,ss.str());
    }
}

void IconEditor::closeTab() {
    if(currentView == NULL)
        return;
    rightTabWidget->removePage(currentView);
    leftTabWidget->removePage(currentView);
    delete currentView;
    currentView = NULL;
}

void IconEditor::refreshThumbnails() {
    currentView->makeThumbnailDir(currentView->dirPath,false);
}

void IconEditor::replaceThumbnails() {
    currentView->makeThumbnailDir(currentView->dirPath,true);
}

void IconEditor::contextMenuEvent( QContextMenuEvent * e) {
    showContextMenu();
    e->accept();
    e->consume();
}

void IconEditor::showContextMenu() {
    if(currentView == NULL)
        return;
    QPopupMenu *contextMenu = new QPopupMenu( this );
    contextMenu->insertItem( "&Undo",  currentView, SLOT(undo()), CTRL+Key_Z );
    contextMenu->insertItem( "Re&do",  currentView, SLOT(redo()), CTRL+SHIFT+Key_Z );
    contextMenu->insertSeparator();
    contextMenu->insertItem( "Cu&t",  currentView, SLOT(cut()), CTRL+Key_X );
    contextMenu->insertItem( "&Copy",  currentView, SLOT(copy()), CTRL+Key_C );
    contextMenu->insertItem( "&Paste",  currentView, SLOT(paste()), CTRL+Key_V );
    contextMenu->insertSeparator();
    contextMenu->insertItem( "Select &All",  currentView, SLOT(selectAllItems()), CTRL+Key_A );
    contextMenu->insertItem( "Pre&view image (double-click)",  currentView, SLOT(showPreview()), CTRL+Key_P );
    if(currentView->dirPath.length() > 0) {
        contextMenu->insertSeparator();
        contextMenu->insertItem( "Create missing thumbnails",  this, SLOT(refreshThumbnails()) );
        contextMenu->insertItem( "Replace all thumbnails",  this, SLOT(replaceThumbnails()) );
        contextMenu->insertSeparator();
        // must be executed here in order to delete the object
        contextMenu->insertItem( "C&lose this window",  this, SLOT(closeTab()) )
        ;
    }

    contextMenu->exec( QCursor::pos() );
    delete contextMenu;
}

#include "iconeditor.moc"

