/***************************************************************************
 *   Copyright (C) 2004 by Paul Lutus                                      *
 *   lutusp@pl-alpha                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "slideprojector.h"
#include "myiconview.h"
#include "myiconviewitem.h"
#include "iconeditor.h"
#include <sys/stat.h>
#include <sys/types.h>

#include <string>
#include <qmessagebox.h>
#include <qprogressbar.h>
#include <qprogressdialog.h>
#include <qclipboard.h>
#include <qeventloop.h>

#include <qimage.h>
#include <iostream>
#include <fstream>
#include <qwhatsthis.h>

//MyIconView::MyIconView()
// : QIconView()
//{
//} KIconView

MyIconView::MyIconView( QWidget * par, IconEditor *p1, SlideProjector *p2,const char *whatsthisText, const char* path,const char* name, WFlags f) :
QIconView( par, name, f ) {
    ieparent = p1;
    spparent = p2;
    mouseOverItem = NULL;
    dirPath = "";
    QWhatsThis::add(this,whatsthisText);
    tcthread = new ThumbCreationThread();

    //timer = new QTimer(this);

    QObject::connect(
        this, SIGNAL(dropped(QDropEvent*,const QValueList<QIconDragItem>&)),
        this, SLOT(slotNewItem(QDropEvent*, const QValueList<QIconDragItem>&))
    );

    QObject::connect(
        this, SIGNAL(moved()),
        this, SLOT(slotMoved())
    );

    QObject::connect(
        this, SIGNAL(onItem(QIconViewItem *)),
        this, SLOT(slotOnItem(QIconViewItem *))
    );

    QObject::connect(
        this, SIGNAL(onViewport()),
        this, SLOT(slotOnViewport())
    );

    QObject::connect(
        this, SIGNAL(doubleClicked ( QIconViewItem *)),
        this, SLOT(showPreview(QIconViewItem *))
    );
    /*
        QObject::connect(
            this, SIGNAL(pressed(QIconViewItem *)),
            this, SLOT(slotPressed(QIconViewItem *))
        );*/

    //setAcceptDrops( TRUE );
    setAutoArrange(true);
    //setShowToolTips(true);
    //QToolTip::add(this,"My ToolTip");
    QFont font( "Helvetica", 1, QFont::Normal );
    setFont( font );
    setResizeMode(QIconView::Adjust);
    setSelectionMode(QIconView::Extended);

    if(path != 0) {
        loadImagesFromDirectory(path);
    }
    setChanged(false);
}

MyIconView::~MyIconView() {
    delete tcthread;
}

QDragObject *MyIconView::dragObject() {

    vector <MyIconViewItem *> v = getSelectedItems();
    string s = makeClipString(v);
    return new QTextDrag( s.c_str(), this );

}

void MyIconView::loadImagesFromDirectory(string path) {
    //printf("path: %s\n",path);
    //fflush(stdout);
    dirPath = path;
    QDir d1(path);
    if ( !d1.exists() ) {
        //qWarning( "Directory %s doesn't exist",path );
    } else {
        d1.setFilter( QDir::Files );
        d1.setSorting(QDir::Name);
        QString tp = d1.path() + "/" + SlideProjector::thumbDirName;
        QDir d2(tp);
        if ( !d2.exists() ) {
            //qWarning( "Directory %s doesn't exist",(const char*)d2.path() );
            queryThumbnailDir(path,true);
        }
        showName = (string) (const char *) d1.dirName();
        clear();
        MyIconViewItem *item;
        for(int i = 0; i < (int) d1.count();i++) {
            string p = d1.path() + "/" + d1[i];
            item = new MyIconViewItem(this,spparent->editIconSize,p);
            if (!item->isValid()) {
                //printf("invalid image: %s\n",(const char*)p);
                //fflush(stdout);
                takeItem(item);
                delete(item);
            }
        }
    }
    setChanged(false);
}

void MyIconView::clearPanel() {
    clear();
    showName = "";
    listFilePath = "";
    setChanged(false);
}

bool MyIconView::loadImagesFromListFile(string ppath,string n) {
    showName = n;
    //printf("load path: %s\n",ppath.c_str());
    //fflush(stdout);
    listFilePath = ppath;
    ifstream ifs(ppath.c_str());
    string path;
    clear();
    MyIconViewItem *item;
    while(getline(ifs,path)) {
        QFileInfo fi(path);
        string ppath = fi.dirPath();
        QDir d1(ppath);
        QString tp = ppath + "/" + SlideProjector::thumbDirName;
        QString tn_full = tp + "/" + SlideProjector::thumbFilePrefix + fi.fileName();
        QFileInfo tfi(tn_full);
        QDir d2(tp);
        if ( !d2.exists() || !tfi.exists() ) {
            queryThumbnailDir(ppath,false);
        }
        item = new MyIconViewItem(this,spparent->editIconSize,path);
        if (!item->isValid()) {
            //printf("invalid image: %s\n",(const char*)tp);
            //fflush(stdout);
            takeItem(item);
            delete(item);

        }
    }
    ifs.close();
    setChanged(false);
    return count() > 0;
}

void MyIconView::saveImagesToListFile() {
    ofstream ofs(listFilePath.c_str());
    for ( MyIconViewItem *item = (MyIconViewItem *) firstItem(); item; item = (MyIconViewItem *) item->nextItem() ) {
        ofs << item->text() << endl;
    }
    ofs.close();
    setChanged(false);
}

void MyIconView::newListFile(string path,string n) {
    showName = n;
    listFilePath = path;
    saveImagesToListFile();
}

void MyIconView::queryThumbnailDir(string dir,bool createAll) {
    string prompt = (string) "There are missing thumbnails for the directory \"" + dir + "\". Create them?";
    int reply = QMessageBox::question(this,spparent->programName + ": Create Thumbnails", prompt
                                      ,QMessageBox::Yes,QMessageBox::No);
    if(reply == QMessageBox::Yes) {
        makeThumbnailDir(dir,createAll);
    }
}

void MyIconView::makeThumbnailDir(string dir,bool createAll) {
    QDir d1(dir);
    QString tp = d1.path() + "/" + SlideProjector::thumbDirName;
    QDir d2(tp);
    tcthread->execute(dir,ieparent->editProgressBar,spparent->editIconSize,createAll);
    // use thread->wait(), not usleep()!
    // much better GUI responsiveness
    while(tcthread->running() && tcthread->wait(100)) {
        qApp->processEvents();
    }
}

bool MyIconView::testOpen() {
    if(listFilePath.length() == 0 && this == ieparent->dest) {
        string prompt = (string)
                        "The show (left) panel is not yet associated with a show file.\n"
                        + " Please choose \"New\" or use the drop-down list\n"
                        + "to select an existing show to edit.";
        QMessageBox::information(this,spparent->programName + ": No file opened",prompt,QMessageBox::Ok);
        return false;
    }
    return true;
}

void MyIconView::slotNewItem( QDropEvent *e, const QValueList<QIconDragItem>& ) {
    if(testOpen()) {
        pushUndo();
        MyIconView *src = (MyIconView *) e->source();
        if(src != 0) {
            vector <MyIconViewItem *> sels = src->getSelectedItems();
            string data = makeClipString(sels);
            // must do this first!
            MyIconViewItem *pi = (MyIconViewItem *) findItem(e->pos());
            // then delete moved items
            //printf("slot new item a\n");
            //fflush(stdout);
            if(e->action() == QDropEvent::Move) {
                src->pushUndo();
                src->deleteItems(sels);
                src->arrangeItemsInGrid(true);
            }
            // re-select the target, it may have been deleted
            //printf("slot new item b\n");
            //fflush(stdout);
	    // check: was "pi" deleted?
            int i = index(pi);
            if(i == -1) {
                pi = (MyIconViewItem *) lastItem();
            }
            //printf("slot new item c = %d\n",(int) pi);
            //fflush(stdout);
            if(count() > 0) { // if count == 0, "pi" is not valid
                setCurrentItem(pi);
                setSelected(pi,true);
            }
            insertItems(data);
            e->acceptAction();
            setChanged(true);

        } else { // src == 0
            if(e->provides("text/plain")) {
                QByteArray qba = e->data("text/plain");
                QString data = qba;
                //printf("src == 0, slot new item: %s\n",(const char *) data);
                //fflush(stdout);
                if ( QTextDrag::decode( e, data ) ) {
                    istringstream ss(data);
                    string path;
                    while(getline(ss,path)) {
                        //printf("slot new item: [%s]\n",(const char *) data);
                        //fflush(stdout);
                        string pref = "file:";
                        if(path.find(pref) != string::npos) {
                            path = path.substr(pref.length());
                        }
                        size_t p = path.find_last_not_of(" \r\n");
                        if(p != string::npos) {
                            path = path.substr(0,p+1);
                        }
                        //printf("src == 0, slot new item: [%s]\n", path.c_str());
                        //fflush(stdout);
                        MyIconViewItem *pi = (MyIconViewItem *) findItem(e->pos());
                        insertIconViewItem(path,pi);
                        setChanged(true);
                    }
                }
                e->acceptAction();
            }
        }
        if(changed) {
            arrangeItemsInGrid(true);
        }
    }
}

void MyIconView::insertIconViewItem(string path, MyIconViewItem *pi) {
    if(testOpen()) {
        //printf("insert item 1\n");
        //fflush(stdout);
        MyIconViewItem *item = new MyIconViewItem(this,spparent->editIconSize, path);
        if(pi != 0) {
            takeItem(item);
            insertItem(item,pi);
            setCurrentItem(item);
            setSelected(item,true);
            setChanged(true);
        } else {
            // item goes to the end by default
            //printf("insertitem pi == 0 %s\n", item->name.c_str());
            //fflush(stdout);
        }
    }
}

void MyIconView::slotMoved() {
    delete currentItem();
    arrangeItemsInGrid(true);
    setChanged(true);
}

void MyIconView::slotOnItem(QIconViewItem *) {
    setCursor(Qt::PointingHandCursor);
}

void MyIconView::slotOnViewport() {
    setCursor(Qt::ArrowCursor );
}
/*
void MyIconView::slotPressed(QIconViewItem* item) {
    //printf("slotPressed %d\n",(int)item);
    //fflush(stdout);
    mouseOverItem = item;
}*/

// we much catch the context menu event
// so we can pass our identity to the parent

void MyIconView::contextMenuEvent( QContextMenuEvent * e) {
    ieparent->currentView = this;
    e->ignore();
    //e->accept();
    //e->consume();
}

void MyIconView::keyPressEvent(QKeyEvent *e) {
    bool accept = false;
    //printf("keypressEvent.\n");
    //fflush(stdout);
    if(e->state() == Qt::ControlButton) {
        //printf("keypressEvent control.\n");
        //fflush(stdout);
        if (e->key() == Qt::Key_C) {
            copy();
            accept = true;
        } else if (e->key() == Qt::Key_X) {
            cut();
            accept = true;
        } else if (e->key() == Qt::Key_V) {
            paste();
            accept = true;
        } else if (e->key() == Qt::Key_A) {
            selectAllItems();
            accept = true;
        } else if (e->key() == Qt::Key_Z) {
            undo();
            accept = true;
        } else if (e->key() == Qt::Key_P) {
            showPreview();
            accept = true;
        }
    } else if (e->state() == Qt::ControlButton|Qt::ShiftButton) {
        if (e->key() == Qt::Key_Z) {
            redo();
            accept = true;
        }
    }
    if(accept) {
        e->accept();
    } else {
        e->ignore();
    }
}

void MyIconView::selectAllItems() {
    selectAll(true);
}

void MyIconView::copy() {
    //printf("copy.\n");
    //fflush(stdout);
    vector <MyIconViewItem *> sels = getSelectedItems();
    string clipData = makeClipString(sels);
    QClipboard *cb = QApplication::clipboard();
    cb->setText(clipData, QClipboard::Clipboard );
}

void MyIconView::cut() {
    //printf("cut.\n");
    //fflush(stdout);
    pushUndo();
    vector <MyIconViewItem *> sels = getSelectedItems();
    string clipData = makeClipString(sels);
    deleteItems(sels);
    setChanged(sels.size() > 0);
    if(changed) {
        arrangeItemsInGrid(true);
    }

    QClipboard *cb = QApplication::clipboard();
    cb->setText(clipData, QClipboard::Clipboard );
}

void MyIconView::paste() {
    if(testOpen()) {
        pushUndo();
        //printf("paste.\n");
        //fflush(stdout);
        QClipboard *cb = QApplication::clipboard();
        string text = (string) (char const *) cb->text(QClipboard::Clipboard);
        insertItems(text);
        setChanged(true);
    }
}

/*void MyIconView::closeWindow() {
    //printf("close.\n");
    //fflush(stdout);
    ieparent->closeTab();
}*/

vector <MyIconViewItem *> MyIconView::getSelectedItems() {
    vector <MyIconViewItem *> sels;
    ostringstream ss;
    for ( MyIconViewItem *item = (MyIconViewItem *) firstItem(); item; item = (MyIconViewItem *) item->nextItem() ) {
        if(item->isSelected()) {
            sels.push_back(item);
        }
    }
    return sels;
}

void MyIconView::deleteItems(vector <MyIconViewItem *> deletes) {
    for(unsigned int i = 0;i < deletes.size();i++) {
        takeItem(deletes[i]);
        delete(deletes[i]);
        setChanged(true);
    }
}

void MyIconView::insertItems(string data) {
    if(data.size() > 0) {
        istringstream ss(data);
        string path;
        while(getline(ss,path)) {
            MyIconViewItem *p = (MyIconViewItem *) currentItem();
            if(p != NULL && !p->isSelected()) {
                p = NULL;
            }
            insertIconViewItem(path,p);
        }
    }
}

string MyIconView::makeClipString(vector <MyIconViewItem *> sels) {
    ostringstream ss;
    for(unsigned int i = 0;i < sels.size();i++) {
        ss << sels[i]->text() << endl;
    }
    return ss.str();
}

void MyIconView::setChanged(bool v) {
    changed = v;
    ieparent->updateSaveStatus(this,changed);
}

void MyIconView::showPreview() {
    showPreview( currentItem());
}

void MyIconView::showPreview(QIconViewItem *item) {
    if(item == 0)
        return;
    spparent->showEditPreviewImage(((MyIconViewItem *)item)->path);
}

void MyIconView::pushUndo() {
    pushUndoRedo(undoVec);
}

void MyIconView::pushRedo() {
    pushUndoRedo(redoVec);
}

void MyIconView::pushUndoRedo(VVS& vec) {
    VS itemv;
    for ( MyIconViewItem *item = (MyIconViewItem *) firstItem(); item; item = (MyIconViewItem *) item->nextItem() ) {
        itemv.push_back(item->path);
    }
    vec.push_back(itemv);
}

void MyIconView::undo() {
    if(undoVec.size() > 0) {
        pushRedo();
        VS v = undoVec.back();
        undoVec.pop_back();
        //redoVec.push_back(v);
        clear();
        for(unsigned int i = 0;i < v.size();i++) {
            new MyIconViewItem(this,spparent->editIconSize, v[i]);
        }
        arrangeItemsInGrid(true);
        setChanged(true);
    } else {
        //printf("undo empty.\n");
        //fflush(stdout);
        QApplication::beep();
    }
}

void MyIconView::redo() {
    if(redoVec.size() > 0) {
        pushUndo();
        VS v = redoVec.back();
        redoVec.pop_back();
        //undoVec.push_back(v);
        clear();
        for(unsigned int i = 0;i < v.size();i++) {
            new MyIconViewItem(this,spparent->editIconSize, v[i]);
        }
        arrangeItemsInGrid(true);
        setChanged(true);
    } else {
        //printf("redo empty.\n");
        //fflush(stdout);
        QApplication::beep();
    }
}
