/***************************************************************************
 *   Copyright (C) 2011 by Paul Lutus                                      *
 *   http://arachnoid.com/administration                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

package com.arachnoid.carpetempus;

import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.SimpleTimeZone;
import java.util.TimeZone;

import android.util.Log;

final public class AlarmEvent extends GenericEvent {
	private static final long serialVersionUID = -1954338135909136032L;

	public AlarmEvent(long wUTCSec, String m) {
		// default alarm time ten minutes in the future
		super(wUTCSec + 600, m);
		recurring = true;
		// default to active
		active = true;
	}

	public AlarmEvent(GenericEvent te) {
		super(te);
	}

	public AlarmEvent copy() {
		return new AlarmEvent(this);
	}

	public String getNameTag() {
		return "Alarm";
	}

	public boolean canRecur() {
		return true;
	}

	public void setSnooze(boolean s) {
		if (s) {
			setSnoozeTimeSec(currentTimeUTCSec() + snoozeTimeValueSec);
			setActive(true);
		} else {
			setSnoozeTimeSec(0);
		}
	}

	public void setActive(boolean state) {
		active = state;
		if (!active) {
			// alarmTripped = false;
			setSnoozeTimeSec(0);
		} else {
			lastAlarmTime = -1;
		}
	}

	public long getNextEventTimeUTCSec() {
		long result = (getSnoozeTimeSec() != 0) ? getSnoozeTimeSec()
				: getEventTimeUTCSec();
		// event time modulo 24 hours
		result %= 86400L;
		long now = currentTimeUTCSec();
		// present time modulo 24 hours
		long now_mod_24 = now % 86400L;
		// only bump time to next day if:
		// 1. event time has passed, or
		// 2. alarm has just sounded
		if (result < now_mod_24
				|| (result == now_mod_24 && lastAlarmTime >= now)) {
			result += 86400L;
		}
		// return the result with any
		// original time > 24 hours added
		result += (now - now_mod_24);
		return result;
	}

	public boolean getActive() {
		return (getEventTimeUTCSec() == 0) ? false : active;
	}

	public void toggleActive() {
		setActive(!active);
	}

	// based on specific date set

	private boolean testDate(GregorianCalendar localCal) {
		if (getSpecDate()) {
			return (getYear() == localCal.get(Calendar.YEAR)
					&& getMonth() == localCal.get(Calendar.MONTH) && getDay() == localCal
						.get(Calendar.DAY_OF_MONTH));
		}
		return true;
	}

	// based on day of month set

	private boolean testDayOfMonth(GregorianCalendar localCal) {
		if (getMonthDays()) {
			// month days numbered 1 ... 31
			int mday = localCal.get(Calendar.DAY_OF_MONTH);
			// convert to 2^(n-1)
			int mask = 1 << (mday - 1);
			int b = getMonthDayBits();
			if ((b & mask) == 0) {
				return false;
			}
		}
		return true;
	}

	// based on day of week set

	private boolean testDayOfWeek(GregorianCalendar localCal) {
		if (getWeekDays()) {
			// weekdays numbered 1 ... 7
			int wday = localCal.get(Calendar.DAY_OF_WEEK);
			// convert to 2^(n-1)
			int mask = 1 << (wday - 1);
			// bits: sunday = 1, monday = 2 ... saturday = 64
			int b = getWeekDayBits();
			if ((b & mask) == 0) {
				return false;
			}
		}
		return true;
	}

	public boolean alarmTime() {
		if (!getActive()) {
			return false;
		}
		long t = currentTimeUTCSec();
		// t += daylightTimeOffset();
		GregorianCalendar localCal = new GregorianCalendar();
		localCal.setTimeInMillis(t * 1000);
		// adjustDaylight(localCal);
		// match both day of week and month
		if (!testDate(localCal) || !testDayOfWeek(localCal)
				|| !testDayOfMonth(localCal)) {
			return false;
		}
		// prevent multiple dialogs
		if (t > lastAlarmTime) {
			// add four seconds to prevent reactivation
			lastAlarmTime = t + 4;
			if (!getRecurring()) {
				setActive(false);
			}
			return true;
		}
		return false;
	}

	protected long daylightTimeOffsetx() {
		Date d = new Date();
		TimeZone tz = TimeZone.getDefault();
		if (tz.inDaylightTime(d)) {
			return tz.getDSTSavings() / 1000;
		} else {
			return 0;
		}
	}

	// read UTC long value, set local time fields
	protected void makeLocalDateTimeFieldsfromEventTime() {
		GregorianCalendar cal = new GregorianCalendar();
		long t = getEventTimeUTCSec();
		// if no date is specified, use today's date and event time
		// t += daylightTimeOffset();
		cal.setTimeInMillis(t * 1000);
		if (!getSpecDate()) {
			adjustDaylight(cal);
		}
		alarmDY = cal.get(Calendar.YEAR);
		alarmDM = cal.get(Calendar.MONTH);
		alarmDDW = cal.get(Calendar.DAY_OF_WEEK);
		alarmDDM = cal.get(Calendar.DAY_OF_MONTH);
		alarmTH = cal.get(Calendar.HOUR_OF_DAY);
		alarmTM = cal.get(Calendar.MINUTE);
		alarmTS = cal.get(Calendar.SECOND);
		// long utch = (t / (60 * 60)) % 24;
		// double localh = cal.get(Calendar.HOUR_OF_DAY);
		// double tz = cal.getTimeZone().getDSTSavings();
		//Log.e("makeLocalDateTimeFieldsfromEventTime", "UTC month: " + alarmDM
		// + ", UTC h: " + utch + "," + localh);
		// TimeZone tz = cal.getTimeZone();
		//Log.e("makeLocalDateTimeFields","Time zone: " + tz);
	}

	// read local time fields, set UTC long value
	public void readEventDateTime() {
		GregorianCalendar gc = new GregorianCalendar();
		gc.set(alarmDY, alarmDM, alarmDDM, alarmTH, alarmTM, alarmTS);
		// adjustDaylight(gc);
		long t = gc.getTimeInMillis() / 1000;
		// t -= daylightTimeOffset();
		setEventTimeUTCSec(t);
	}

	protected String secToTimeString(CarpeTempusApplication app) {
		GregorianCalendar gc = new GregorianCalendar();
		long t = getEventTimeUTCSec();
		// t += daylightTimeOffset();
		gc.setTimeInMillis(t * 1000);
		adjustDaylight(gc);
		// this snooze asterisk ruins the layout's appearance
		String s = ""; //(getSnoozeTimeSec() != 0) ? "* " : "";
		if (app.get24HourFormat()) {
			s += String.format("%02d:%02d:%02d", gc.get(Calendar.HOUR_OF_DAY),
					gc.get(Calendar.MINUTE), gc.get(Calendar.SECOND));
		} else {
			int hour = gc.get(Calendar.HOUR_OF_DAY);
			int ampm = (hour >= 12) ? 1 : 0;
			hour = hour % 12;
			hour = (hour == 0) ? 12 : hour;
			s += String.format("%02d:%02d:%02d %s", hour,
					gc.get(Calendar.MINUTE), gc.get(Calendar.SECOND),
					ampmStrings[ampm]);
		}
		return s;
	}

	public String toString() {
		long t = getEventTimeUTCSec();
		if (!getWeekDays() && !getMonthDays()) {
			t %= 86400;
		}
		return "AlarmEvent: \"" + getMessage() + "\" , "
				+ dtStringFromTimeMS(t * 1000);
	}

}
